'use client'

import { useState, useEffect, useCallback } from 'react'
import {
    Plus, Search, Edit3, X, Check, ShieldCheck, Users2,
    ToggleLeft, ToggleRight, Loader2, Eye, EyeOff, Phone, Crown
} from 'lucide-react'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import EmptyState from '@/components/EmptyState'
import { useToast } from '@/components/Toast'

interface DBUser {
    id: string
    nama: string
    email: string
    role: string
    noWa: string
    isActive: boolean
    targetPsbMonthly: number
    targetVisitDaily: number
    createdAt: string
}

const roles = [
    { value: 'super_admin', label: 'Super Admin', color: '#DC2626', bg: '#FEE2E2' },
    { value: 'admin', label: 'Admin', color: '#EF4444', bg: '#FEE2E2' },
    { value: 'spv_marketing', label: 'SPV Marketing', color: '#7C3AED', bg: '#EDE9FE' },
    { value: 'sales_senior', label: 'Sales Senior', color: '#2563EB', bg: '#DBEAFE' },
    { value: 'sales_junior', label: 'Sales Junior', color: '#059669', bg: '#D1FAE5' },
]

function getRoleBadge(role: string) {
    const r = roles.find(rl => rl.value === role) || roles[4]
    return (
        <span style={{
            display: 'inline-flex', padding: '2px 8px', borderRadius: 'var(--radius-full)',
            fontSize: 'var(--text-xs)', fontWeight: 600, color: r.color, background: r.bg,
        }}>
            {r.label}
        </span>
    )
}

export default function AdminUsersPage() {
    const { addToast } = useToast()

    const [users, setUsers] = useState<DBUser[]>([])
    const [loading, setLoading] = useState(true)
    const [saving, setSaving] = useState(false)
    const [showForm, setShowForm] = useState(false)
    const [editId, setEditId] = useState<string | null>(null)
    const [search, setSearch] = useState('')
    const [showPassword, setShowPassword] = useState(false)
    const [form, setForm] = useState({
        nama: '', email: '', password: '', role: 'sales_junior', noWa: '',
        targetPsbMonthly: 15, targetVisitDaily: 10,
    })

    const fetchUsers = useCallback(async () => {
        try {
            const res = await fetch('/api/users')
            if (res.ok) {
                const data = await res.json()
                setUsers(data)
            }
        } catch (err) {
            console.error('Failed to fetch users:', err)
        } finally {
            setLoading(false)
        }
    }, [])

    useEffect(() => {
        fetchUsers()
    }, [fetchUsers])

    const filteredUsers = users.filter(u =>
        !search || u.nama.toLowerCase().includes(search.toLowerCase()) || u.email.toLowerCase().includes(search.toLowerCase())
    )

    const handleSubmit = async () => {
        if (!form.nama || !form.email || !form.noWa) {
            addToast('error', 'Data belum lengkap', 'Nama, email, dan nomor WA wajib diisi.')
            return
        }

        if (!editId && !form.password) {
            addToast('error', 'Password wajib diisi', 'Password diperlukan untuk user baru.')
            return
        }

        setSaving(true)

        try {
            if (editId) {
                // Update existing user
                const res = await fetch(`/api/users/${editId}`, {
                    method: 'PUT',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        nama: form.nama,
                        email: form.email,
                        role: form.role,
                        noWa: form.noWa,
                        targetPsbMonthly: form.targetPsbMonthly,
                        targetVisitDaily: form.targetVisitDaily,
                    }),
                })
                if (res.ok) {
                    addToast('success', 'User diperbarui', form.nama)
                    fetchUsers()
                } else {
                    const err = await res.json()
                    addToast('error', 'Gagal update', err.error || 'Terjadi kesalahan')
                }
            } else {
                // Create new user
                const res = await fetch('/api/users', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        nama: form.nama,
                        email: form.email,
                        password: form.password,
                        role: form.role,
                        noWa: form.noWa,
                        targetPsbMonthly: form.targetPsbMonthly,
                        targetVisitDaily: form.targetVisitDaily,
                    }),
                })
                if (res.ok) {
                    addToast('success', 'User ditambahkan', form.nama)
                    fetchUsers()
                } else {
                    const err = await res.json()
                    addToast('error', 'Gagal menambahkan', err.error || 'Terjadi kesalahan')
                }
            }
        } catch {
            addToast('error', 'Error', 'Tidak dapat terhubung ke server')
        }

        setSaving(false)
        setShowForm(false)
        setEditId(null)
        setForm({ nama: '', email: '', password: '', role: 'sales_junior', noWa: '', targetPsbMonthly: 15, targetVisitDaily: 10 })
    }

    const startEdit = (user: DBUser) => {
        setForm({
            nama: user.nama, email: user.email, password: '', role: user.role,
            noWa: user.noWa, targetPsbMonthly: user.targetPsbMonthly, targetVisitDaily: user.targetVisitDaily,
        })
        setEditId(user.id)
        setShowForm(true)
    }

    const toggleActive = async (userId: string) => {
        try {
            const res = await fetch(`/api/users/${userId}`, { method: 'PATCH' })
            if (res.ok) {
                fetchUsers()
                addToast('success', 'Status diperbarui', '')
            }
        } catch {
            addToast('error', 'Gagal', 'Tidak dapat mengubah status')
        }
    }

    if (loading) {
        return (
            <AnimatedPage className="flex items-center justify-center" style={{ minHeight: '60vh' }}>
                <Loader2 size={32} className="animate-spin" style={{ color: 'var(--color-accent-500)' }} />
            </AnimatedPage>
        )
    }

    return (
        <AnimatedPage className="space-y-4">
            <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-3">
                <div>
                    <div className="flex items-center gap-2">
                        <Crown size={24} style={{ color: 'var(--color-accent-500)' }} />
                        <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>User Management</h1>
                    </div>
                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>{users.length} pengguna terdaftar di database</p>
                </div>
                <button onClick={() => { setShowForm(true); setEditId(null); setForm({ nama: '', email: '', password: '', role: 'sales_junior', noWa: '', targetPsbMonthly: 15, targetVisitDaily: 10 }) }} className="btn-accent">
                    <Plus size={16} /> Tambah User
                </button>
            </div>

            {/* Search */}
            <div className="relative">
                <Search size={16} style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: 'var(--color-muted)' }} />
                <input
                    className="form-input"
                    placeholder="Cari nama atau email..."
                    value={search}
                    onChange={(e) => setSearch(e.target.value)}
                    style={{ paddingLeft: 36 }}
                />
            </div>

            {/* User List */}
            {users.length === 0 ? (
                <EmptyState
                    icon={<Users2 size={32} />}
                    title="Belum Ada User"
                    description="Tambahkan user baru untuk mulai mengelola tim sales Anda."
                    actionLabel="+ Tambah User"
                    onAction={() => setShowForm(true)}
                />
            ) : (
                <div className="space-y-2">
                    {filteredUsers.map((user, i) => (
                        <StaggeredItem key={user.id} index={i}>
                            <div className="card card-hover-lift" style={{ padding: '1rem' }}>
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center gap-3">
                                        <div style={{
                                            width: 40, height: 40, borderRadius: 'var(--radius-full)',
                                            background: user.isActive ? 'var(--color-accent-100)' : 'var(--color-subtle)',
                                            display: 'flex', alignItems: 'center', justifyContent: 'center',
                                            fontSize: 'var(--text-sm)', fontWeight: 700,
                                            color: user.isActive ? 'var(--color-accent-500)' : 'var(--color-muted)',
                                        }}>
                                            {user.nama.split(' ').map(w => w[0]).join('').slice(0, 2).toUpperCase()}
                                        </div>
                                        <div>
                                            <div className="flex items-center gap-2 flex-wrap">
                                                <h3 style={{ fontSize: 'var(--text-sm)', fontWeight: 700 }}>{user.nama}</h3>
                                                {getRoleBadge(user.role)}
                                                {!user.isActive && (
                                                    <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-danger)', fontWeight: 600 }}>Nonaktif</span>
                                                )}
                                            </div>
                                            <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>{user.email} · {user.noWa}</p>
                                            <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginTop: 2 }}>
                                                Target PSB: {user.targetPsbMonthly}/bln · Visit: {user.targetVisitDaily}/hari
                                            </p>
                                        </div>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <button onClick={() => toggleActive(user.id)}
                                            style={{ background: 'none', border: 'none', cursor: 'pointer', color: user.isActive ? 'var(--color-success)' : 'var(--color-muted)' }}>
                                            {user.isActive ? <ToggleRight size={20} /> : <ToggleLeft size={20} />}
                                        </button>
                                        <button onClick={() => startEdit(user)}
                                            style={{ background: 'none', border: 'none', cursor: 'pointer', color: 'var(--color-muted)' }}>
                                            <Edit3 size={14} />
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </StaggeredItem>
                    ))}
                </div>
            )}

            {/* Add/Edit Form Modal */}
            {showForm && (
                <div className="fixed inset-0 z-50 flex items-center justify-center p-4" style={{ background: 'rgba(0,0,0,0.4)' }}>
                    <div className="card card-enter" style={{ width: '100%', maxWidth: 480, padding: '1.5rem', maxHeight: '90vh', overflow: 'auto' }}>
                        <div className="flex items-center justify-between mb-4">
                            <h3 style={{ fontWeight: 700, fontSize: 'var(--text-lg)' }}>{editId ? 'Edit User' : 'Tambah User Baru'}</h3>
                            <button onClick={() => setShowForm(false)} style={{ background: 'none', border: 'none', cursor: 'pointer' }}><X size={18} /></button>
                        </div>
                        <div className="space-y-3">
                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                                <div className="sm:col-span-2">
                                    <label className="form-label">Nama Lengkap *</label>
                                    <input className="form-input" value={form.nama} onChange={e => setForm(p => ({ ...p, nama: e.target.value }))} required />
                                </div>
                                <div>
                                    <label className="form-label">Email *</label>
                                    <input className="form-input" type="email" value={form.email} onChange={e => setForm(p => ({ ...p, email: e.target.value }))} required />
                                </div>
                                <div>
                                    <label className="form-label">No. WhatsApp *</label>
                                    <input className="form-input" value={form.noWa} onChange={e => setForm(p => ({ ...p, noWa: e.target.value }))} required />
                                </div>
                                {!editId && (
                                    <div className="sm:col-span-2">
                                        <label className="form-label">Password *</label>
                                        <div className="relative">
                                            <input
                                                className="form-input"
                                                type={showPassword ? 'text' : 'password'}
                                                value={form.password}
                                                onChange={e => setForm(p => ({ ...p, password: e.target.value }))}
                                                placeholder="Min. 6 karakter"
                                                required
                                                style={{ paddingRight: 40 }}
                                            />
                                            <button
                                                type="button"
                                                onClick={() => setShowPassword(!showPassword)}
                                                style={{ position: 'absolute', right: 8, top: '50%', transform: 'translateY(-50%)', background: 'none', border: 'none', cursor: 'pointer', color: 'var(--color-muted)' }}
                                            >
                                                {showPassword ? <EyeOff size={16} /> : <Eye size={16} />}
                                            </button>
                                        </div>
                                    </div>
                                )}
                                <div>
                                    <label className="form-label">Role</label>
                                    <select className="form-input form-select" value={form.role} onChange={e => setForm(p => ({ ...p, role: e.target.value }))}>
                                        {roles.map(r => <option key={r.value} value={r.value}>{r.label}</option>)}
                                    </select>
                                </div>
                                <div>
                                    <label className="form-label">Target PSB/Bulan</label>
                                    <input className="form-input" type="number" value={form.targetPsbMonthly} onChange={e => setForm(p => ({ ...p, targetPsbMonthly: parseInt(e.target.value) || 0 }))} />
                                </div>
                                <div>
                                    <label className="form-label">Target Visit/Hari</label>
                                    <input className="form-input" type="number" value={form.targetVisitDaily} onChange={e => setForm(p => ({ ...p, targetVisitDaily: parseInt(e.target.value) || 0 }))} />
                                </div>
                            </div>
                            <button onClick={handleSubmit} className="btn-accent" disabled={saving} style={{ width: '100%' }}>
                                {saving ? <Loader2 size={14} className="animate-spin" /> : <Check size={14} />}
                                {editId ? ' Simpan Perubahan' : ' Tambah User'}
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </AnimatedPage>
    )
}
