'use client'

import { useState, useMemo } from 'react'
import {
    Plus, Users, Wifi, Target, ChevronDown, ChevronUp,
    MapPin, Edit3, X, Check, Search, Inbox
} from 'lucide-react'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import { useCRMStore } from '@/store/crmStore'
import EmptyState from '@/components/EmptyState'
import { useToast } from '@/components/Toast'
import { ALL_KECAMATAN } from '@/lib/utils'

function getStatusColor(psb: number, target: number) {
    if (target <= 0) return 'var(--color-muted)'
    const pct = (psb / target) * 100
    if (pct >= 80) return 'var(--color-success)'
    if (pct >= 50) return 'var(--color-warning)'
    return 'var(--color-danger)'
}

export default function AreasPage() {
    const [search, setSearch] = useState('')
    const [expandedId, setExpandedId] = useState<string | null>(null)

    const leads = useCRMStore(s => s.leads)
    const { addToast } = useToast()

    // Derive area data from leads
    const areaData = useMemo(() => {
        const areaMap: Record<string, { kecamatan: string; leads: number; psb: number; kelurahans: Set<string> }> = {}

        leads.forEach(l => {
            if (!areaMap[l.kecamatan]) {
                areaMap[l.kecamatan] = { kecamatan: l.kecamatan, leads: 0, psb: 0, kelurahans: new Set() }
            }
            areaMap[l.kecamatan].leads++
            if (l.pipelineStage === 'won') areaMap[l.kecamatan].psb++
            if (l.kelurahan) areaMap[l.kecamatan].kelurahans.add(l.kelurahan)
        })

        // Also include kecamatan from master list without leads
        ALL_KECAMATAN.forEach(k => {
            if (!areaMap[k]) {
                areaMap[k] = { kecamatan: k, leads: 0, psb: 0, kelurahans: new Set() }
            }
        })

        return Object.values(areaMap)
            .filter(a => !search || a.kecamatan.toLowerCase().includes(search.toLowerCase()))
            .sort((a, b) => b.leads - a.leads)
    }, [leads, search])

    const totalLeads = leads.length
    const totalPsb = leads.filter(l => l.pipelineStage === 'won').length

    return (
        <AnimatedPage className="space-y-4">
            <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-3">
                <div>
                    <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>Area Coverage</h1>
                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>
                        {areaData.length} kecamatan · {totalLeads} leads · {totalPsb} PSB
                    </p>
                </div>
            </div>

            {/* Search */}
            <div className="relative">
                <Search size={16} style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: 'var(--color-muted)' }} />
                <input
                    className="form-input"
                    placeholder="Cari kecamatan..."
                    value={search}
                    onChange={(e) => setSearch(e.target.value)}
                    style={{ paddingLeft: 36 }}
                />
            </div>

            {/* Summary Cards */}
            <div className="grid grid-cols-3 gap-3">
                <StaggeredItem index={0}>
                    <div className="card" style={{ textAlign: 'center', padding: '1rem' }}>
                        <div style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>{totalLeads}</div>
                        <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>Total Leads</div>
                    </div>
                </StaggeredItem>
                <StaggeredItem index={1}>
                    <div className="card" style={{ textAlign: 'center', padding: '1rem' }}>
                        <div style={{ fontSize: 'var(--text-2xl)', fontWeight: 700, color: 'var(--color-success)' }}>{totalPsb}</div>
                        <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>Total PSB</div>
                    </div>
                </StaggeredItem>
                <StaggeredItem index={2}>
                    <div className="card" style={{ textAlign: 'center', padding: '1rem' }}>
                        <div style={{ fontSize: 'var(--text-2xl)', fontWeight: 700, color: 'var(--color-accent-500)' }}>
                            {areaData.filter(a => a.leads > 0).length}
                        </div>
                        <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>Area Aktif</div>
                    </div>
                </StaggeredItem>
            </div>

            {/* Area List */}
            {areaData.length === 0 ? (
                <EmptyState
                    icon={<MapPin size={32} />}
                    title="Belum Ada Data Area"
                    description="Data area akan muncul otomatis saat Anda menambahkan lead."
                    actionLabel="+ Tambah Lead Baru"
                    actionHref="/leads/new"
                />
            ) : (
                <div className="space-y-2">
                    {areaData.map((area, i) => (
                        <StaggeredItem key={area.kecamatan} index={i + 3}>
                            <div className="card card-hover-lift" style={{ padding: '1rem', cursor: 'pointer' }}
                                onClick={() => setExpandedId(expandedId === area.kecamatan ? null : area.kecamatan)}>
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center gap-3">
                                        <div style={{ width: 40, height: 40, borderRadius: 'var(--radius-md)', background: area.leads > 0 ? '#EFF6FF' : 'var(--color-subtle)', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
                                            <MapPin size={18} style={{ color: area.leads > 0 ? '#3B82F6' : 'var(--color-muted)' }} />
                                        </div>
                                        <div>
                                            <h3 style={{ fontSize: 'var(--text-base)', fontWeight: 700 }}>{area.kecamatan}</h3>
                                            <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                {area.leads} leads · {area.psb} PSB
                                            </p>
                                        </div>
                                    </div>
                                    {expandedId === area.kecamatan ? <ChevronUp size={16} /> : <ChevronDown size={16} />}
                                </div>
                                {expandedId === area.kecamatan && (
                                    <div style={{ marginTop: '0.75rem', paddingTop: '0.75rem', borderTop: '1px solid var(--color-border)' }}>
                                        {area.kelurahans.size > 0 ? (
                                            <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                <strong>Kelurahan:</strong> {[...area.kelurahans].join(', ')}
                                            </div>
                                        ) : (
                                            <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>Belum ada lead di area ini.</p>
                                        )}
                                    </div>
                                )}
                            </div>
                        </StaggeredItem>
                    ))}
                </div>
            )}
        </AnimatedPage>
    )
}
