'use client'

import { useState, useEffect, useMemo } from 'react'
import Link from 'next/link'
import {
    UserPlus, Wifi, Target, MapPin, TrendingUp, TrendingDown,
    Bell, ChevronDown, Clock, Phone, MessageCircle, CheckCircle,
    Eye, Plus, X, ArrowUpRight, Award, Search, Inbox
} from 'lucide-react'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import { useCountUp } from '@/hooks/useCountUp'
import { useCRMStore } from '@/store/crmStore'
import { useLeadAccess } from '@/hooks/useLeadAccess'

// ── Helpers ──────────────────────────────────────────────
function getGreeting() {
    const h = new Date().getHours()
    if (h < 12) return 'Selamat Pagi'
    if (h < 15) return 'Selamat Siang'
    if (h < 18) return 'Selamat Sore'
    return 'Selamat Malam'
}

function formatRp(v: number) {
    return 'Rp ' + v.toLocaleString('id-ID')
}

const activityIcons: Record<string, any> = {
    visit: { icon: Eye, color: '#8B5CF6', bg: '#F5F3FF' },
    phone_call: { icon: Phone, color: '#3B82F6', bg: '#EFF6FF' },
    wa_chat: { icon: MessageCircle, color: '#10B981', bg: '#ECFDF5' },
    follow_up: { icon: Clock, color: '#F59E0B', bg: '#FFFBEB' },
    survey_lokasi: { icon: MapPin, color: '#E8712B', bg: '#FFF3E0' },
    registrasi_psb: { icon: CheckCircle, color: '#10B981', bg: '#ECFDF5' },
}

// ── KPI Card with animated counter ─────────────────────
function KPICard({ label, value, target, icon: Icon, color, bg, index }: {
    label: string; value: number; target: number;
    icon: any; color: string; bg: string; index: number;
}) {
    const animatedValue = useCountUp(value, 1000, index * 100)
    const pct = target > 0 ? Math.min((value / target) * 100, 100) : 0

    return (
        <StaggeredItem index={index}>
            <div className="kpi-card card-hover-lift">
                <div className="flex items-center justify-between mb-3">
                    <div className="kpi-icon-circle" style={{ background: bg }}>
                        <Icon size={18} style={{ color }} />
                    </div>
                    {target > 0 && (
                        <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', fontWeight: 600 }}>
                            Target: {target}
                        </span>
                    )}
                </div>
                <div className="kpi-value kpi-value-animated" style={{ marginBottom: 4 }}>
                    {animatedValue}
                </div>
                <div className="kpi-label">{label}</div>
                {target > 0 && (
                    <div className="kpi-progress" style={{ marginTop: 8 }}>
                        <div className="kpi-progress-fill" style={{
                            width: `${pct}%`,
                            background: pct >= 80 ? 'var(--color-success)' : pct >= 50 ? 'var(--color-warning)' : color,
                        }} />
                    </div>
                )}
            </div>
        </StaggeredItem>
    )
}

// ── Component ──────────────────────────────────────────────
export default function DashboardPage() {
    const [showNotif, setShowNotif] = useState(false)
    const [fabOpen, setFabOpen] = useState(false)

    const allLeads = useCRMStore(s => s.leads)
    const activities = useCRMStore(s => s.activities)
    const { filterLeads, userName } = useLeadAccess()

    const leads = useMemo(() => filterLeads(allLeads), [allLeads, filterLeads])

    // Compute KPIs from real data
    const kpis = useMemo(() => {
        const now = new Date()
        const thisMonth = leads.filter(l => {
            const d = new Date(l.createdAt)
            return d.getMonth() === now.getMonth() && d.getFullYear() === now.getFullYear()
        })
        const won = leads.filter(l => l.pipelineStage === 'won')
        const wonThisMonth = won.filter(l => {
            const d = new Date(l.updatedAt)
            return d.getMonth() === now.getMonth() && d.getFullYear() === now.getFullYear()
        })
        const todayStr = now.toISOString().split('T')[0]
        const todayActivities = activities.filter(a => a.activityDate.startsWith(todayStr))

        return [
            { label: 'PSB Bulan Ini', value: wonThisMonth.length, target: 50, icon: Wifi, color: '#10B981', bg: '#ECFDF5' },
            { label: 'New Lead', value: thisMonth.length, target: 0, icon: UserPlus, color: '#E8712B', bg: '#FFF3E0' },
            { label: 'Total Leads', value: leads.filter(l => l.pipelineStage !== 'lost').length, target: 0, icon: Target, color: '#3B82F6', bg: '#EFF6FF' },
            { label: 'Visit Hari Ini', value: todayActivities.filter(a => a.activityType === 'visit').length, target: 10, icon: MapPin, color: '#8B5CF6', bg: '#F5F3FF' },
        ]
    }, [leads, activities])

    // Pipeline summary
    const stageCounts = useMemo(() => {
        const stages = ['new_lead', 'contacted', 'visited', 'surveyed', 'negotiation', 'won', 'lost']
        return stages.map(key => ({
            key,
            label: key === 'new_lead' ? 'New Lead' : key === 'won' ? 'Won ✅' : key === 'lost' ? 'Lost ❌' : key.charAt(0).toUpperCase() + key.slice(1),
            count: leads.filter(l => l.pipelineStage === key).length,
            value: leads.filter(l => l.pipelineStage === key).reduce((sum, l) => sum + (l.estimatedValue || 0), 0),
        }))
    }, [leads])

    const hasLeads = leads.length > 0
    const hasActivities = activities.length > 0

    // Get current user
    const displayName = userName || 'User'

    return (
        <AnimatedPage className="space-y-6">
            {/* ── TOP BAR ───────────────────────────────── */}
            <div className="top-bar">
                <div className="top-bar-left">
                    <h1 className="top-bar-greeting">{getGreeting()}, {displayName} 👋</h1>
                    <p className="top-bar-summary">
                        {hasLeads
                            ? `${kpis[0].value} PSB bulan ini, ${leads.filter(l => l.pipelineStage !== 'won' && l.pipelineStage !== 'lost').length} leads aktif.`
                            : 'Mulai tambahkan lead pertama Anda!'}
                    </p>
                </div>
                <div className="top-bar-right">
                    {/* Notification Bell */}
                    <div className="relative">
                        <button
                            className="notification-bell"
                            onClick={() => setShowNotif(!showNotif)}
                            style={{ height: 44, width: 44, minWidth: 44, minHeight: 44 }}
                        >
                            <Bell size={18} />
                        </button>

                        {showNotif && (
                            <div className="notification-panel">
                                <div className="notification-panel-header">
                                    <span style={{ fontWeight: 700, fontSize: 'var(--text-lg)' }}>Notifikasi</span>
                                </div>
                                <div className="notification-list" style={{ padding: '2rem', textAlign: 'center' }}>
                                    <Inbox size={32} style={{ color: 'var(--color-muted)', margin: '0 auto 0.5rem' }} />
                                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>Belum ada notifikasi</p>
                                </div>
                            </div>
                        )}
                    </div>

                    {/* User Avatar */}
                    <div className="user-avatar hidden md:flex">
                        {userName.split(' ').map(w => w[0]).join('').slice(0, 2).toUpperCase()}
                    </div>
                </div>
            </div>

            {/* ── KPI CARDS ─────────────────────────────── */}
            <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 md:gap-4">
                {kpis.map((kpi, i) => (
                    <KPICard key={kpi.label} {...kpi} index={i} />
                ))}
            </div>

            {/* ── PIPELINE OVERVIEW ────────────────────── */}
            <StaggeredItem index={5}>
                <div className="card" style={{ padding: 0, overflow: 'hidden' }}>
                    <div className="flex items-center justify-between p-4 pb-3" style={{ borderBottom: '1px solid var(--color-border)' }}>
                        <h3 style={{ fontSize: 'var(--text-lg)', fontWeight: 700 }}>Pipeline Overview</h3>
                        <Link href="/pipeline" className="btn-ghost" style={{ fontSize: 'var(--text-xs)', color: 'var(--color-accent-500)' }}>
                            Lihat Detail →
                        </Link>
                    </div>
                    {hasLeads ? (
                        <div className="pipeline-scroll-wrapper" style={{ padding: '1rem', overflowX: 'auto' }}>
                            <div style={{ display: 'flex', gap: '0.5rem', minWidth: 'max-content' }}>
                                {stageCounts.map(s => (
                                    <div key={s.key} style={{
                                        padding: '0.75rem 1rem', borderRadius: 'var(--radius-md)',
                                        background: 'var(--color-subtle)', minWidth: 120, flex: '1 0 auto',
                                    }}>
                                        <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', fontWeight: 600, marginBottom: 4 }}>{s.label}</div>
                                        <div style={{ fontSize: 'var(--text-xl)', fontWeight: 700 }}>{s.count}</div>
                                        {s.value > 0 && (
                                            <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginTop: 2 }}>{formatRp(s.value)}</div>
                                        )}
                                    </div>
                                ))}
                            </div>
                        </div>
                    ) : (
                        <div style={{ padding: '3rem 2rem', textAlign: 'center' }}>
                            <div className="empty-icon-pulse" style={{ display: 'inline-block' }}>
                                <Target size={36} style={{ color: 'var(--color-muted)' }} />
                            </div>
                            <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)', marginTop: '0.75rem' }}>
                                Pipeline kosong. Tambahkan lead untuk mulai tracking.
                            </p>
                            <Link href="/leads/new" className="btn-accent" style={{ marginTop: '1rem', display: 'inline-flex' }}>
                                <Plus size={16} /> Tambah Lead
                            </Link>
                        </div>
                    )}
                </div>
            </StaggeredItem>

            {/* ── BOTTOM SECTION: (Activity Feed) ──────── */}
            <StaggeredItem index={6}>
                <div className="card" style={{ padding: 0, overflow: 'hidden', display: 'flex', flexDirection: 'column' }}>
                    <div className="flex items-center justify-between p-4 pb-3">
                        <div className="flex items-center gap-2">
                            <h3 style={{ fontSize: 'var(--text-lg)', fontWeight: 700 }}>Aktivitas Terbaru</h3>
                            {hasActivities && <span className="animate-pulse-live" style={{ width: 6, height: 6, borderRadius: '50%', background: 'var(--color-success)' }} />}
                        </div>
                    </div>
                    {hasActivities ? (
                        <div className="activity-feed-scroll" style={{ flex: 1, overflowY: 'auto', maxHeight: 'min(400px, 50dvh)', padding: '0 1rem 1rem' }}>
                            {activities.slice(0, 10).map((act, i) => {
                                const iconInfo = activityIcons[act.activityType] || activityIcons.visit
                                const ActIcon = iconInfo.icon
                                return (
                                    <StaggeredItem key={act.id} index={i}>
                                        <div className="timeline-item">
                                            <div className="timeline-dot" style={{ borderColor: iconInfo.color, background: iconInfo.bg }}>
                                                <ActIcon size={10} style={{ color: iconInfo.color }} />
                                            </div>
                                            <div>
                                                <p style={{ fontSize: 'var(--text-sm)', fontWeight: 600, color: 'var(--color-text)' }}>
                                                    {act.activityType === 'visit' ? 'Visit' : act.activityType === 'phone_call' ? 'Telepon' : act.activityType === 'wa_chat' ? 'WA Chat' : act.activityType} ke {act.leadName}
                                                </p>
                                                <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginTop: 2 }}>
                                                    {act.sales} · {act.area} · {new Date(act.activityDate).toLocaleDateString('id-ID')}
                                                </p>
                                            </div>
                                        </div>
                                    </StaggeredItem>
                                )
                            })}
                        </div>
                    ) : (
                        <div style={{ padding: '3rem 2rem', textAlign: 'center' }}>
                            <div className="empty-icon-pulse" style={{ display: 'inline-block' }}>
                                <Clock size={36} style={{ color: 'var(--color-muted)' }} />
                            </div>
                            <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)', marginTop: '0.75rem' }}>
                                Belum ada aktivitas tercatat.
                            </p>
                        </div>
                    )}
                </div>
            </StaggeredItem>

            {/* ── FAB MENU ──────────────────────────────── */}
            <div className="fab-container">
                {fabOpen && (
                    <>
                        <Link href="/leads/new" className="fab-mini">
                            <span className="fab-mini-icon" style={{ background: 'var(--color-accent-100)', color: 'var(--color-accent-500)' }}>
                                <UserPlus size={16} />
                            </span>
                            Lead Baru
                        </Link>
                    </>
                )}
                <button
                    className={`fab-button ${fabOpen ? 'open' : ''}`}
                    onClick={() => setFabOpen(!fabOpen)}
                >
                    <Plus size={24} />
                </button>
            </div>

            {/* Click outside to close */}
            {showNotif && <div className="fixed inset-0 z-40" onClick={() => setShowNotif(false)} style={{ background: 'transparent' }} />}
            {fabOpen && <div className="fixed inset-0 z-30" onClick={() => setFabOpen(false)} style={{ background: 'transparent' }} />}
        </AnimatedPage>
    )
}
