'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import {
    ArrowLeft, MapPin, Check, Loader2, Plus, Navigation,
} from 'lucide-react'
import Link from 'next/link'
import AnimatedPage from '@/components/AnimatedPage'
import { useCRMStore } from '@/store/crmStore'
import { useToast } from '@/components/Toast'
import { ALL_KECAMATAN, SUMBER_LEAD_OPTIONS, PRIORITY_OPTIONS, SEGMEN_OPTIONS } from '@/lib/utils'

const paketOptions = [
    { nama: 'Galaksi', speed: '10 Mbps', harga: 189000 },
    { nama: 'Bintang', speed: '20 Mbps', harga: 219000 },
    { nama: 'Nebula', speed: '50 Mbps', harga: 279000 },
    { nama: 'Supernova', speed: '100 Mbps', harga: 399000 },
    { nama: 'Custom Corporate', speed: 'Dedicated', harga: 0 },
    { nama: 'Mitra Package', speed: 'Wholesale', harga: 0 },
]

export default function NewLeadPage() {
    const router = useRouter()
    const addLead = useCRMStore(s => s.addLead)
    const { addToast } = useToast()

    const [loading, setLoading] = useState(false)
    const [gpsLoading, setGpsLoading] = useState(false)
    const [success, setSuccess] = useState(false)
    const [form, setForm] = useState({
        namaLengkap: '', noWa: '', alamat: '',
        kelurahan: '', kecamatan: '', latitude: undefined as number | undefined,
        longitude: undefined as number | undefined,
        segmen: 'retail', sumberLead: 'canvassing',
        paketInterest: '', priority: 'cold',
        estimatedValue: undefined as number | undefined,
        notes: '', nextFollowupDate: '',
    })

    const handleChange = (field: string, value: string) => {
        setForm(prev => ({ ...prev, [field]: value }))
    }

    const captureGPS = () => {
        if (!navigator.geolocation) {
            addToast('error', 'GPS tidak tersedia', 'Browser Anda tidak mendukung geolokasi.')
            return
        }
        setGpsLoading(true)
        navigator.geolocation.getCurrentPosition(
            async (pos) => {
                const lat = pos.coords.latitude
                const lon = pos.coords.longitude
                setForm(prev => ({ ...prev, latitude: lat, longitude: lon }))

                // Reverse geocoding via Nominatim (OpenStreetMap) - free, no API key
                try {
                    const res = await fetch(
                        `https://nominatim.openstreetmap.org/reverse?lat=${lat}&lon=${lon}&format=json&addressdetails=1&accept-language=id`,
                        { headers: { 'User-Agent': 'JABNET-CRM/2.0' } }
                    )
                    if (res.ok) {
                        const data = await res.json()
                        const addr = data.address || {}

                        // Build alamat from road/neighbourhood/hamlet
                        const alamatParts = [addr.road, addr.hamlet, addr.neighbourhood].filter(Boolean)
                        const alamat = alamatParts.length > 0 ? alamatParts.join(', ') : (data.display_name?.split(',').slice(0, 2).join(',') || '')

                        // Kelurahan = village / suburb / quarter
                        const kelurahan = addr.village || addr.suburb || addr.quarter || ''

                        // Kecamatan = city_district / county / municipality
                        const rawKecamatan = addr.city_district || addr.county || addr.municipality || ''
                        // Try to match with known kecamatan list (strip "Kecamatan" prefix if needed)
                        const cleanKec = rawKecamatan.replace(/^Kecamatan\s+/i, '').trim()
                        const matchedKec = ALL_KECAMATAN.find(k =>
                            k.toLowerCase() === cleanKec.toLowerCase() ||
                            cleanKec.toLowerCase().includes(k.toLowerCase()) ||
                            k.toLowerCase().includes(cleanKec.toLowerCase())
                        )

                        setForm(prev => ({
                            ...prev,
                            alamat: prev.alamat || alamat,
                            kelurahan: prev.kelurahan || kelurahan,
                            kecamatan: prev.kecamatan || (matchedKec || ''),
                        }))

                        addToast('success', 'Lokasi ditangkap!',
                            `📍 ${lat.toFixed(6)}, ${lon.toFixed(6)}${kelurahan ? ` · ${kelurahan}` : ''}${matchedKec ? ` · ${matchedKec}` : ''}`)
                    } else {
                        addToast('success', 'Lokasi ditangkap!', `${lat.toFixed(6)}, ${lon.toFixed(6)} (geocoding gagal)`)
                    }
                } catch {
                    addToast('success', 'Lokasi ditangkap!', `${lat.toFixed(6)}, ${lon.toFixed(6)} (geocoding gagal)`)
                }

                setGpsLoading(false)
            },
            (err) => {
                setGpsLoading(false)
                addToast('error', 'Gagal menangkap GPS', err.message)
            },
            { enableHighAccuracy: true, timeout: 10000 }
        )
    }

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault()
        if (!form.namaLengkap || !form.noWa || !form.kecamatan) {
            addToast('error', 'Data belum lengkap', 'Nama, No WA, dan Kecamatan wajib diisi.')
            return
        }

        setLoading(true)

        // Select paket and calculate estimated value
        const selectedPaket = paketOptions.find(p => p.nama === form.paketInterest)
        const estimatedValue = selectedPaket?.harga || form.estimatedValue

        let userName = 'Admin'
        try {
            const u = JSON.parse(localStorage.getItem('jabnet_user') || '{}')
            userName = u.nama || 'Admin'
        } catch { }

        setTimeout(() => {
            addLead({
                namaLengkap: form.namaLengkap,
                noWa: form.noWa,
                alamat: form.alamat,
                kelurahan: form.kelurahan,
                kecamatan: form.kecamatan,
                latitude: form.latitude,
                longitude: form.longitude,
                segmen: form.segmen,
                sumberLead: form.sumberLead,
                paketInterest: form.paketInterest,
                priority: form.priority,
                pipelineStage: 'new_lead',
                estimatedValue,
                notes: form.notes,
                nextFollowupDate: form.nextFollowupDate,
                assignedSales: userName,
                createdBy: userName,
            })
            setLoading(false)
            setSuccess(true)
            addToast('success', 'Lead berhasil ditambahkan!', form.namaLengkap)
        }, 500)
    }

    const handleAddAnother = () => {
        setSuccess(false)
        setForm({
            namaLengkap: '', noWa: '', alamat: '',
            kelurahan: '', kecamatan: '', latitude: undefined,
            longitude: undefined,
            segmen: 'retail', sumberLead: 'canvassing',
            paketInterest: '', priority: 'cold',
            estimatedValue: undefined,
            notes: '', nextFollowupDate: '',
        })
    }

    // Success screen
    if (success) {
        return (
            <AnimatedPage className="max-w-xl mx-auto" style={{ paddingTop: '4rem', textAlign: 'center' }}>
                <div style={{ width: 64, height: 64, borderRadius: '50%', background: '#ECFDF5', display: 'flex', alignItems: 'center', justifyContent: 'center', margin: '0 auto 1.5rem' }}>
                    <Check size={28} style={{ color: '#10B981' }} />
                </div>
                <h2 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700, marginBottom: 8 }}>Lead Berhasil Ditambahkan!</h2>
                <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)', marginBottom: '2rem' }}>
                    {form.namaLengkap} telah ditambahkan ke pipeline.
                </p>
                <div className="flex items-center justify-center gap-3">
                    <button onClick={handleAddAnother} className="btn-outline">
                        <Plus size={16} /> Tambah Lagi
                    </button>
                    <Link href="/leads" className="btn-accent">
                        Lihat Leads
                    </Link>
                </div>
            </AnimatedPage>
        )
    }

    return (
        <AnimatedPage className="max-w-2xl mx-auto">
            {/* Header */}
            <div className="flex items-center gap-3 mb-6">
                <Link href="/leads" style={{ color: 'var(--color-muted)', display: 'flex' }}>
                    <ArrowLeft size={20} />
                </Link>
                <div>
                    <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>Input Lead Baru</h1>
                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>Isi data calon pelanggan</p>
                </div>
            </div>

            <form onSubmit={handleSubmit} className="space-y-4">
                {/* Data Kontak */}
                <div className="card" style={{ padding: '1.5rem' }}>
                    <h3 style={{ fontSize: 'var(--text-lg)', fontWeight: 700, marginBottom: '1rem' }}>📋 Data Kontak</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div className="md:col-span-2">
                            <label className="form-label">Nama Lengkap *</label>
                            <input className="form-input" placeholder="Nama calon pelanggan" value={form.namaLengkap} onChange={(e) => handleChange('namaLengkap', e.target.value)} required />
                        </div>
                        <div>
                            <label className="form-label">No. WhatsApp *</label>
                            <input className="form-input" placeholder="628xxxxxxxxxx" value={form.noWa} onChange={(e) => handleChange('noWa', e.target.value)} required />
                        </div>
                        <div>
                            <label className="form-label">Segmen</label>
                            <select className="form-input form-select" value={form.segmen} onChange={(e) => handleChange('segmen', e.target.value)}>
                                {SEGMEN_OPTIONS.map(s => <option key={s.value} value={s.value}>{s.label}</option>)}
                            </select>
                        </div>
                    </div>
                </div>

                {/* Lokasi */}
                <div className="card" style={{ padding: '1.5rem' }}>
                    <h3 style={{ fontSize: 'var(--text-lg)', fontWeight: 700, marginBottom: '1rem' }}>📍 Lokasi</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div className="md:col-span-2">
                            <label className="form-label">Alamat</label>
                            <input className="form-input" placeholder="Jl. / Gg. / Komp." value={form.alamat} onChange={(e) => handleChange('alamat', e.target.value)} />
                        </div>
                        <div>
                            <label className="form-label">Kecamatan *</label>
                            <select className="form-input form-select" value={form.kecamatan} onChange={(e) => handleChange('kecamatan', e.target.value)} required>
                                <option value="">Pilih Kecamatan</option>
                                {ALL_KECAMATAN.map(k => <option key={k} value={k}>{k}</option>)}
                            </select>
                        </div>
                        <div>
                            <label className="form-label">Kelurahan</label>
                            <input className="form-input" placeholder="Kelurahan/Desa" value={form.kelurahan} onChange={(e) => handleChange('kelurahan', e.target.value)} />
                        </div>
                        <div className="md:col-span-2">
                            <button type="button" onClick={captureGPS} className="btn-outline" disabled={gpsLoading} style={{ width: '100%' }}>
                                {gpsLoading ? <Loader2 size={14} className="animate-spin" /> : <Navigation size={14} />}
                                {gpsLoading ? 'Menangkap GPS...' : form.latitude ? `📍 ${form.latitude.toFixed(6)}, ${form.longitude?.toFixed(6)}` : 'Tangkap Lokasi GPS'}
                            </button>
                        </div>
                    </div>
                </div>

                {/* Interest */}
                <div className="card" style={{ padding: '1.5rem' }}>
                    <h3 style={{ fontSize: 'var(--text-lg)', fontWeight: 700, marginBottom: '1rem' }}>💰 Interest & Prioritas</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label className="form-label">Sumber Lead</label>
                            <select className="form-input form-select" value={form.sumberLead} onChange={(e) => handleChange('sumberLead', e.target.value)}>
                                {SUMBER_LEAD_OPTIONS.map(s => <option key={s.value} value={s.value}>{s.label}</option>)}
                            </select>
                        </div>
                        <div>
                            <label className="form-label">Paket Interest</label>
                            <select className="form-input form-select" value={form.paketInterest} onChange={(e) => {
                                handleChange('paketInterest', e.target.value)
                                const p = paketOptions.find(pk => pk.nama === e.target.value)
                                if (p && p.harga > 0) setForm(prev => ({ ...prev, estimatedValue: p.harga }))
                            }}>
                                <option value="">Pilih Paket</option>
                                {paketOptions.map(p => <option key={p.nama} value={p.nama}>{p.nama} ({p.speed}) {p.harga > 0 ? `- Rp ${p.harga.toLocaleString()}` : ''}</option>)}
                            </select>
                        </div>
                        <div>
                            <label className="form-label">Prioritas</label>
                            <select className="form-input form-select" value={form.priority} onChange={(e) => handleChange('priority', e.target.value)}>
                                {PRIORITY_OPTIONS.map(p => <option key={p.value} value={p.value}>{p.label}</option>)}
                            </select>
                        </div>
                        <div>
                            <label className="form-label">Follow-up Berikutnya</label>
                            <input type="date" className="form-input" value={form.nextFollowupDate} onChange={(e) => handleChange('nextFollowupDate', e.target.value)} />
                        </div>
                        <div className="md:col-span-2">
                            <label className="form-label">Catatan</label>
                            <textarea className="form-input" rows={3} placeholder="Catatan tambahan..." value={form.notes} onChange={(e) => handleChange('notes', e.target.value)} />
                        </div>
                    </div>
                </div>

                {/* Submit */}
                <button type="submit" className="btn-accent" disabled={loading} style={{ width: '100%', padding: '0.875rem', fontSize: 'var(--text-base)', fontWeight: 700 }}>
                    {loading ? <><Loader2 size={16} className="animate-spin" /> Menyimpan...</> : <><Check size={16} /> Simpan Lead</>}
                </button>
            </form>
        </AnimatedPage>
    )
}
