'use client'

import { useState, useCallback, useMemo } from 'react'
import Link from 'next/link'
import {
    Search, Plus, ChevronRight, MapPin, Phone, Clock,
    Calendar, Flame, Snowflake, Sun, List, LayoutGrid,
    Filter, ArrowRight, MoreHorizontal, Eye, Inbox
} from 'lucide-react'
import { getStageLabel, getStageColor, formatDate } from '@/lib/utils'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import { useCRMStore } from '@/store/crmStore'
import EmptyState from '@/components/EmptyState'
import { useLeadAccess } from '@/hooks/useLeadAccess'

const kanbanStages = [
    { value: 'new_lead', label: 'New Lead', color: '#3B82F6' },
    { value: 'contacted', label: 'Contacted', color: '#6366F1' },
    { value: 'visited', label: 'Visited', color: '#8B5CF6' },
    { value: 'surveyed', label: 'Surveyed', color: '#F59E0B' },
    { value: 'negotiation', label: 'Negotiation', color: '#E8712B' },
    { value: 'won', label: 'Won', color: '#10B981' },
    { value: 'lost', label: 'Lost', color: '#64748B' },
]

function PriorityBadge({ priority }: { priority: string }) {
    const config: Record<string, { icon: any; color: string; bg: string; label: string }> = {
        hot: { icon: Flame, color: '#EF4444', bg: '#FEF2F2', label: 'Hot' },
        warm: { icon: Sun, color: '#F59E0B', bg: '#FFFBEB', label: 'Warm' },
        cold: { icon: Snowflake, color: '#64748B', bg: '#F1F5F9', label: 'Cold' },
    }
    const c = config[priority] || config.cold
    const Icon = c.icon
    return (
        <span style={{ display: 'inline-flex', alignItems: 'center', gap: 3, padding: '2px 8px', borderRadius: 'var(--radius-full)', background: c.bg, color: c.color, fontSize: 'var(--text-xs)', fontWeight: 600 }}>
            <Icon size={10} /> {c.label}
        </span>
    )
}

function formatRp(v: number) { return 'Rp ' + v.toLocaleString('id-ID') }

export default function LeadsPage() {
    const [viewMode, setViewMode] = useState<'list' | 'kanban'>('list')
    const [activeFilter, setActiveFilter] = useState('semua')
    const [searchQuery, setSearchQuery] = useState('')

    const allLeads = useCRMStore(s => s.leads)
    const moveLead = useCRMStore(s => s.moveLead)
    const { filterLeads, isSupervisor, userName } = useLeadAccess()

    // Apply role-based visibility filter
    const leads = useMemo(() => filterLeads(allLeads), [allLeads, filterLeads])

    const todayStr = new Date().toISOString().split('T')[0]

    const tabs = useMemo(() => [
        { id: 'semua', label: 'Semua', count: leads.length },
        { id: 'followup', label: 'Follow-up Hari Ini', count: leads.filter(l => l.nextFollowupDate === todayStr).length },
        { id: 'hot', label: 'Hot 🔥', count: leads.filter(l => l.priority === 'hot' && l.pipelineStage !== 'won' && l.pipelineStage !== 'lost').length },
    ], [leads, todayStr])

    const filteredLeads = useMemo(() => leads.filter(lead => {
        if (activeFilter === 'followup') return lead.nextFollowupDate === todayStr
        if (activeFilter === 'hot') return lead.priority === 'hot' && lead.pipelineStage !== 'won' && lead.pipelineStage !== 'lost'
        if (searchQuery) {
            const q = searchQuery.toLowerCase()
            return lead.namaLengkap.toLowerCase().includes(q) || lead.noWa.includes(q) || lead.kecamatan.toLowerCase().includes(q)
        }
        return true
    }), [leads, activeFilter, searchQuery, todayStr])

    const moveLeadToStage = useCallback((leadId: string, newStage: string) => {
        moveLead(leadId, newStage)
    }, [moveLead])

    return (
        <AnimatedPage className="space-y-4">
            {/* Header */}
            <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-3">
                <div>
                    <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>My Leads</h1>
                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>
                        {leads.length} leads {!isSupervisor && `· Milik ${userName}`}{isSupervisor && '· Semua Tim'}
                    </p>
                </div>
                <div className="flex items-center gap-2">
                    {/* View Toggle */}
                    <div style={{ display: 'flex', border: '1px solid var(--color-border)', borderRadius: 'var(--radius-md)', overflow: 'hidden' }}>
                        <button
                            onClick={() => setViewMode('list')}
                            style={{ padding: '6px 12px', background: viewMode === 'list' ? 'var(--color-primary-900)' : 'var(--color-card)', color: viewMode === 'list' ? 'white' : 'var(--color-muted)', border: 'none', cursor: 'pointer', display: 'flex', alignItems: 'center', gap: 4, fontSize: 'var(--text-xs)', fontWeight: 600 }}
                        >
                            <List size={14} /> List
                        </button>
                        <button
                            onClick={() => setViewMode('kanban')}
                            style={{ padding: '6px 12px', background: viewMode === 'kanban' ? 'var(--color-primary-900)' : 'var(--color-card)', color: viewMode === 'kanban' ? 'white' : 'var(--color-muted)', border: 'none', borderLeft: '1px solid var(--color-border)', cursor: 'pointer', display: 'flex', alignItems: 'center', gap: 4, fontSize: 'var(--text-xs)', fontWeight: 600 }}
                        >
                            <LayoutGrid size={14} /> Kanban
                        </button>
                    </div>
                    <Link href="/leads/new" className="btn-accent">
                        <Plus size={16} /> <span className="hidden sm:inline">Tambah Lead</span>
                    </Link>
                </div>
            </div>

            {/* Search */}
            <div className="relative">
                <Search size={16} style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: 'var(--color-muted)' }} />
                <input
                    className="form-input"
                    placeholder="Cari nama, no WA, kecamatan..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    style={{ paddingLeft: 36 }}
                />
            </div>

            {/* Filter Tabs */}
            <div className="tab-bar">
                {tabs.map(t => (
                    <button key={t.id} className={`tab-item ${activeFilter === t.id ? 'active' : ''}`} onClick={() => setActiveFilter(t.id)}>
                        {t.label} <span style={{ marginLeft: 4, opacity: 0.6 }}>({t.count})</span>
                    </button>
                ))}
            </div>

            {/* Content */}
            {leads.length === 0 ? (
                <EmptyState
                    icon={<Inbox size={32} />}
                    title="Belum Ada Lead"
                    description="Mulai tambahkan lead pertama Anda dengan klik tombol di bawah."
                    actionLabel="+ Tambah Lead Baru"
                    actionHref="/leads/new"
                />
            ) : viewMode === 'list' ? (
                <div className="space-y-2">
                    {filteredLeads.length === 0 ? (
                        <div style={{ padding: '3rem 2rem', textAlign: 'center' }}>
                            <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>Tidak ada lead yang cocok dengan filter.</p>
                        </div>
                    ) : (
                        filteredLeads.map((lead, i) => (
                            <StaggeredItem key={lead.id} index={i}>
                                <Link href={`/leads/${lead.id}`} className="card card-hover-lift" style={{ display: 'block', textDecoration: 'none', padding: '0.875rem 1rem' }}>
                                    <div className="flex items-start justify-between gap-3">
                                        <div style={{ flex: 1, minWidth: 0 }}>
                                            <div className="flex items-center gap-2 mb-1">
                                                <h3 style={{ fontSize: 'var(--text-base)', fontWeight: 700, color: 'var(--color-text)' }}>{lead.namaLengkap}</h3>
                                                <PriorityBadge priority={lead.priority} />
                                            </div>
                                            <div className="flex flex-wrap items-center gap-x-3 gap-y-1" style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                <span className="flex items-center gap-1"><MapPin size={10} /> {lead.kecamatan}</span>
                                                <span className="flex items-center gap-1"><Phone size={10} /> {lead.noWa}</span>
                                                {lead.paketInterest && <span>Paket: {lead.paketInterest}</span>}
                                                {isSupervisor && lead.assignedSales && (
                                                    <span style={{ background: '#EFF6FF', color: '#2563EB', padding: '1px 6px', borderRadius: 'var(--radius-full)', fontWeight: 600 }}>
                                                        👤 {lead.assignedSales}
                                                    </span>
                                                )}
                                            </div>
                                        </div>
                                        <div className="flex flex-col items-end gap-1" style={{ flexShrink: 0 }}>
                                            <span style={{
                                                fontSize: 'var(--text-xs)', fontWeight: 600, padding: '2px 8px',
                                                borderRadius: 'var(--radius-full)', color: getStageColor(lead.pipelineStage),
                                                background: getStageColor(lead.pipelineStage) + '18',
                                            }}>
                                                {getStageLabel(lead.pipelineStage)}
                                            </span>
                                            {lead.estimatedValue && (
                                                <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', fontWeight: 600 }}>
                                                    {formatRp(lead.estimatedValue)}
                                                </span>
                                            )}
                                        </div>
                                    </div>
                                </Link>
                            </StaggeredItem>
                        ))
                    )}
                </div>
            ) : (
                /* KANBAN VIEW */
                <div className="kanban-board">
                    {kanbanStages.map(stage => {
                        const stageLeads = filteredLeads.filter(l => l.pipelineStage === stage.value)
                        return (
                            <div key={stage.value} className="kanban-column" style={{ width: 280 }}>
                                <div className="kanban-column-header">
                                    <div className="flex items-center justify-between">
                                        <div className="flex items-center gap-2">
                                            <div style={{ width: 8, height: 8, borderRadius: '50%', background: stage.color }} />
                                            <span style={{ fontSize: 'var(--text-sm)', fontWeight: 700 }}>{stage.label}</span>
                                        </div>
                                        <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', fontWeight: 600 }}>{stageLeads.length}</span>
                                    </div>
                                </div>
                                <div style={{ padding: '0.5rem', overflowY: 'auto', flex: 1 }}>
                                    {stageLeads.length === 0 ? (
                                        <div style={{ padding: '2rem 0.5rem', textAlign: 'center', fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                            Kosong
                                        </div>
                                    ) : (
                                        stageLeads.map(lead => (
                                            <Link key={lead.id} href={`/leads/${lead.id}`} style={{ display: 'block', textDecoration: 'none', marginBottom: '0.5rem' }}>
                                                <div className="card card-hover-lift" style={{ padding: '0.75rem' }}>
                                                    <p style={{ fontSize: 'var(--text-sm)', fontWeight: 600, color: 'var(--color-text)', marginBottom: 4 }}>{lead.namaLengkap}</p>
                                                    <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                        <span className="flex items-center gap-1"><MapPin size={10} /> {lead.kecamatan}</span>
                                                    </div>
                                                    <div className="flex items-center justify-between" style={{ marginTop: 6 }}>
                                                        <PriorityBadge priority={lead.priority} />
                                                        {lead.estimatedValue && <span style={{ fontSize: 'var(--text-xs)', fontWeight: 600, color: 'var(--color-muted)' }}>{formatRp(lead.estimatedValue!)}</span>}
                                                    </div>
                                                    {lead.assignedSales && (
                                                        <div style={{ marginTop: 4, fontSize: 'var(--text-xs)', color: '#2563EB', fontWeight: 600 }}>
                                                            👤 {lead.assignedSales}
                                                        </div>
                                                    )}
                                                </div>
                                            </Link>
                                        ))
                                    )}
                                </div>
                            </div>
                        )
                    })}
                </div>
            )}
        </AnimatedPage>
    )
}
