'use client'

import { useState, useMemo } from 'react'
import Link from 'next/link'
import {
    Search, Filter, Plus, X, ChevronDown, Clock,
    Phone, MapPin, MessageCircle, Eye, UserPlus,
    ArrowUpDown, LayoutGrid, List, Table2,
    ChevronRight, AlertTriangle, Calendar, Inbox
} from 'lucide-react'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import { useCRMStore } from '@/store/crmStore'
import EmptyState from '@/components/EmptyState'
import { useLeadAccess } from '@/hooks/useLeadAccess'

const stageConfig = [
    { key: 'new_lead', label: 'New Lead', color: '#3B82F6', icon: '📥', maxDays: 3 },
    { key: 'contacted', label: 'Contacted', color: '#6366F1', icon: '📞', maxDays: 5 },
    { key: 'visited', label: 'Visited', color: '#8B5CF6', icon: '📍', maxDays: 7 },
    { key: 'surveyed', label: 'Surveyed', color: '#F59E0B', icon: '📋', maxDays: 7 },
    { key: 'negotiation', label: 'Negotiation', color: '#E8712B', icon: '🤝', maxDays: 14 },
    { key: 'won', label: 'Won ✅', color: '#10B981', icon: '✅', maxDays: 0 },
    { key: 'lost', label: 'Lost ❌', color: '#64748B', icon: '❌', maxDays: 0 },
]

function formatRp(v: number) { return 'Rp ' + v.toLocaleString('id-ID') }

export default function PipelinePage() {
    const [searchQuery, setSearchQuery] = useState('')
    const [selectedStage, setSelectedStage] = useState<string | null>(null)

    const allLeads = useCRMStore(s => s.leads)
    const moveLead = useCRMStore(s => s.moveLead)
    const { filterLeads, isSupervisor } = useLeadAccess()

    const leads = useMemo(() => filterLeads(allLeads), [allLeads, filterLeads])

    const getStageLeads = (stageKey: string) => {
        return leads.filter(l => {
            if (l.pipelineStage !== stageKey) return false
            if (searchQuery) {
                const q = searchQuery.toLowerCase()
                return l.namaLengkap.toLowerCase().includes(q) || l.kecamatan.toLowerCase().includes(q)
            }
            return true
        })
    }

    // Total pipeline value
    const totalValue = useMemo(() => {
        return leads
            .filter(l => l.pipelineStage !== 'won' && l.pipelineStage !== 'lost')
            .reduce((sum, l) => sum + (l.estimatedValue || 0), 0)
    }, [leads])

    return (
        <AnimatedPage className="space-y-4">
            {/* Header */}
            <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-3">
                <div>
                    <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>Pipeline</h1>
                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>
                        {leads.filter(l => l.pipelineStage !== 'won' && l.pipelineStage !== 'lost').length} leads aktif
                        {totalValue > 0 && ` · Total: ${formatRp(totalValue)}`}
                    </p>
                </div>
                <Link href="/leads/new" className="btn-accent">
                    <Plus size={16} /> Tambah Lead
                </Link>
            </div>

            {/* Search */}
            <div className="relative">
                <Search size={16} style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: 'var(--color-muted)' }} />
                <input
                    className="form-input"
                    placeholder="Cari nama atau kecamatan..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    style={{ paddingLeft: 36 }}
                />
            </div>

            {leads.length === 0 ? (
                <EmptyState
                    icon={<Inbox size={32} />}
                    title="Pipeline Kosong"
                    description="Tambahkan lead pertama Anda untuk mulai tracking pipeline penjualan."
                    actionLabel="+ Tambah Lead Baru"
                    actionHref="/leads/new"
                />
            ) : (
                /* Kanban Board */
                <div className="kanban-board">
                    {stageConfig.map((stage, si) => {
                        const stageLeads = getStageLeads(stage.key)
                        const stageValue = stageLeads.reduce((sum, l) => sum + (l.estimatedValue || 0), 0)
                        return (
                            <StaggeredItem key={stage.key} index={si}>
                                <div className="kanban-column" style={{ width: 280 }}>
                                    <div className="kanban-column-header">
                                        <div className="flex items-center justify-between mb-1">
                                            <div className="flex items-center gap-2">
                                                <span>{stage.icon}</span>
                                                <span style={{ fontSize: 'var(--text-sm)', fontWeight: 700 }}>{stage.label}</span>
                                            </div>
                                            <span style={{
                                                fontSize: 'var(--text-xs)', fontWeight: 700, padding: '1px 6px',
                                                borderRadius: 'var(--radius-full)', background: stage.color + '18', color: stage.color,
                                            }}>{stageLeads.length}</span>
                                        </div>
                                        {stageValue > 0 && (
                                            <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>{formatRp(stageValue)}</p>
                                        )}
                                    </div>
                                    <div style={{ padding: '0.5rem', overflowY: 'auto', flex: 1 }}>
                                        {stageLeads.length === 0 ? (
                                            <div style={{ padding: '2rem 0.5rem', textAlign: 'center', fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                Kosong
                                            </div>
                                        ) : (
                                            stageLeads.map((lead, i) => {
                                                const daysInStage = Math.floor((Date.now() - new Date(lead.pipelineStageDate).getTime()) / 86400000)
                                                const isOverdue = stage.maxDays > 0 && daysInStage > stage.maxDays
                                                return (
                                                    <Link key={lead.id} href={`/leads/${lead.id}`} style={{ display: 'block', textDecoration: 'none', marginBottom: '0.5rem' }}>
                                                        <div className="card card-hover-lift" style={{ padding: '0.75rem', borderLeft: `3px solid ${stage.color}` }}>
                                                            <div className="flex items-start justify-between mb-1">
                                                                <p style={{ fontSize: 'var(--text-sm)', fontWeight: 600, color: 'var(--color-text)' }}>{lead.namaLengkap}</p>
                                                                {isOverdue && <AlertTriangle size={12} style={{ color: 'var(--color-danger)', flexShrink: 0 }} />}
                                                            </div>
                                                            <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                                                <span className="flex items-center gap-1"><MapPin size={10} /> {lead.kecamatan}</span>
                                                            </div>
                                                            <div className="flex items-center justify-between" style={{ marginTop: 6 }}>
                                                                <span style={{ fontSize: 'var(--text-xs)', color: isOverdue ? 'var(--color-danger)' : 'var(--color-muted)' }}>
                                                                    <Clock size={10} style={{ display: 'inline', marginRight: 2 }} />
                                                                    {daysInStage}d
                                                                </span>
                                                                {lead.estimatedValue && (
                                                                    <span style={{ fontSize: 'var(--text-xs)', fontWeight: 600 }}>{formatRp(lead.estimatedValue)}</span>
                                                                )}
                                                            </div>
                                                            {lead.assignedSales && (
                                                                <div style={{ marginTop: 4, fontSize: 'var(--text-xs)', color: '#2563EB', fontWeight: 600 }}>
                                                                    👤 {lead.assignedSales}
                                                                </div>
                                                            )}
                                                        </div>
                                                    </Link>
                                                )
                                            })
                                        )}
                                    </div>
                                </div>
                            </StaggeredItem>
                        )
                    })}
                </div>
            )}
        </AnimatedPage>
    )
}
