'use client'

import { useState, useMemo } from 'react'
import {
    Calendar, Download, TrendingUp, TrendingDown,
    Users, MapPin, Target, Wifi, Eye, Phone,
    MessageCircle, Award, BarChart3, GitBranch, Inbox
} from 'lucide-react'
import AnimatedPage, { StaggeredItem } from '@/components/AnimatedPage'
import { useCRMStore } from '@/store/crmStore'
import EmptyState from '@/components/EmptyState'
import { useCountUp } from '@/hooks/useCountUp'
import { useLeadAccess } from '@/hooks/useLeadAccess'

const reportTabs = [
    { key: 'ringkasan', label: '📊 Ringkasan' },
    { key: 'funnel', label: '🔄 Funnel' },
    { key: 'area', label: '📍 Per Area' },
]

const stageLabels: Record<string, string> = {
    new_lead: 'New Lead', contacted: 'Contacted', visited: 'Visited',
    surveyed: 'Surveyed', negotiation: 'Negotiation', won: 'Won', lost: 'Lost',
}

const stageColors: Record<string, string> = {
    new_lead: '#3B82F6', contacted: '#6366F1', visited: '#8B5CF6',
    surveyed: '#F59E0B', negotiation: '#E8712B', won: '#10B981', lost: '#64748B',
}

export default function ReportsPage() {
    const [activeTab, setActiveTab] = useState('ringkasan')

    const allLeads = useCRMStore(s => s.leads)
    const activities = useCRMStore(s => s.activities)
    const { filterLeads } = useLeadAccess()

    const leads = useMemo(() => filterLeads(allLeads), [allLeads, filterLeads])

    const hasData = leads.length > 0

    // Stats
    const totalLeads = leads.length
    const wonLeads = leads.filter(l => l.pipelineStage === 'won').length
    const lostLeads = leads.filter(l => l.pipelineStage === 'lost').length
    const activeLeads = totalLeads - wonLeads - lostLeads
    const conversionRate = totalLeads > 0 ? Math.round((wonLeads / totalLeads) * 100) : 0
    const totalValue = leads.filter(l => l.pipelineStage === 'won').reduce((s, l) => s + (l.estimatedValue || 0), 0)

    const animatedTotal = useCountUp(totalLeads, 800)
    const animatedWon = useCountUp(wonLeads, 800, 100)
    const animatedRate = useCountUp(conversionRate, 800, 200)

    // Funnel data
    const funnelData = useMemo(() => {
        const stages = ['new_lead', 'contacted', 'visited', 'surveyed', 'negotiation', 'won']
        return stages.map(key => ({
            stage: stageLabels[key],
            count: leads.filter(l => l.pipelineStage === key).length,
            color: stageColors[key],
        }))
    }, [leads])

    const maxFunnelCount = Math.max(...funnelData.map(f => f.count), 1)

    // Area data
    const areaStats = useMemo(() => {
        const map: Record<string, { area: string; leads: number; psb: number }> = {}
        leads.forEach(l => {
            if (!map[l.kecamatan]) map[l.kecamatan] = { area: l.kecamatan, leads: 0, psb: 0 }
            map[l.kecamatan].leads++
            if (l.pipelineStage === 'won') map[l.kecamatan].psb++
        })
        return Object.values(map).sort((a, b) => b.leads - a.leads)
    }, [leads])

    return (
        <AnimatedPage className="space-y-4">
            <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-3">
                <div>
                    <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700 }}>Laporan</h1>
                    <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>Performa penjualan dan analisis</p>
                </div>
            </div>

            {/* Tabs */}
            <div className="tab-bar">
                {reportTabs.map(t => (
                    <button key={t.key} className={`tab-item ${activeTab === t.key ? 'active' : ''}`} onClick={() => setActiveTab(t.key)}>
                        {t.label}
                    </button>
                ))}
            </div>

            {!hasData ? (
                <EmptyState
                    icon={<BarChart3 size={32} />}
                    title="Belum Ada Data"
                    description="Laporan akan tersedia setelah Anda menambahkan lead dan mencatat aktivitas."
                    actionLabel="+ Tambah Lead"
                    actionHref="/leads/new"
                />
            ) : (
                <>
                    {/* RINGKASAN */}
                    {activeTab === 'ringkasan' && (
                        <div className="space-y-4">
                            <div className="grid grid-cols-2 lg:grid-cols-4 gap-3">
                                <StaggeredItem index={0}>
                                    <div className="kpi-card card-hover-lift">
                                        <div className="kpi-label">Total Leads</div>
                                        <div className="kpi-value" style={{ marginTop: 4 }}>{animatedTotal}</div>
                                    </div>
                                </StaggeredItem>
                                <StaggeredItem index={1}>
                                    <div className="kpi-card card-hover-lift">
                                        <div className="kpi-label">PSB (Won)</div>
                                        <div className="kpi-value" style={{ marginTop: 4, color: 'var(--color-success)' }}>{animatedWon}</div>
                                    </div>
                                </StaggeredItem>
                                <StaggeredItem index={2}>
                                    <div className="kpi-card card-hover-lift">
                                        <div className="kpi-label">Conversion Rate</div>
                                        <div className="kpi-value" style={{ marginTop: 4 }}>{animatedRate}%</div>
                                    </div>
                                </StaggeredItem>
                                <StaggeredItem index={3}>
                                    <div className="kpi-card card-hover-lift">
                                        <div className="kpi-label">Total Revenue</div>
                                        <div className="kpi-value" style={{ marginTop: 4, fontSize: 'var(--text-xl)' }}>
                                            Rp {totalValue.toLocaleString('id-ID')}
                                        </div>
                                    </div>
                                </StaggeredItem>
                            </div>

                            {/* Quick Stats */}
                            <StaggeredItem index={4}>
                                <div className="card" style={{ padding: '1.25rem' }}>
                                    <h3 style={{ fontSize: 'var(--text-base)', fontWeight: 700, marginBottom: '0.75rem' }}>Status Pipeline</h3>
                                    <div className="space-y-2">
                                        {Object.entries(stageLabels).map(([key, label]) => {
                                            const count = leads.filter(l => l.pipelineStage === key).length
                                            const pct = totalLeads > 0 ? (count / totalLeads) * 100 : 0
                                            return (
                                                <div key={key} className="flex items-center gap-3" style={{ fontSize: 'var(--text-sm)' }}>
                                                    <div style={{ width: 8, height: 8, borderRadius: '50%', background: stageColors[key], flexShrink: 0 }} />
                                                    <span style={{ width: 100, fontWeight: 500 }}>{label}</span>
                                                    <div style={{ flex: 1, height: 6, background: 'var(--color-subtle)', borderRadius: 3, overflow: 'hidden' }}>
                                                        <div style={{ width: `${pct}%`, height: '100%', background: stageColors[key], borderRadius: 3, transition: 'width 0.8s ease' }} />
                                                    </div>
                                                    <span style={{ width: 30, textAlign: 'right', fontWeight: 700, fontSize: 'var(--text-xs)' }}>{count}</span>
                                                </div>
                                            )
                                        })}
                                    </div>
                                </div>
                            </StaggeredItem>
                        </div>
                    )}

                    {/* FUNNEL */}
                    {activeTab === 'funnel' && (
                        <StaggeredItem index={0}>
                            <div className="card" style={{ padding: '1.5rem' }}>
                                <h3 style={{ fontSize: 'var(--text-lg)', fontWeight: 700, marginBottom: '1.5rem' }}>Sales Funnel</h3>
                                <div className="space-y-3">
                                    {funnelData.map((f, i) => (
                                        <div key={f.stage} className="flex items-center gap-3">
                                            <span style={{ width: 90, fontSize: 'var(--text-sm)', fontWeight: 500, flexShrink: 0 }}>{f.stage}</span>
                                            <div style={{ flex: 1, height: 28, background: 'var(--color-subtle)', borderRadius: 'var(--radius-md)', overflow: 'hidden', position: 'relative' }}>
                                                <div style={{
                                                    width: `${(f.count / maxFunnelCount) * 100}%`,
                                                    height: '100%', background: f.color, borderRadius: 'var(--radius-md)',
                                                    transition: 'width 0.8s cubic-bezier(0.22, 1, 0.36, 1)',
                                                    transitionDelay: `${i * 0.1}s`,
                                                    display: 'flex', alignItems: 'center', justifyContent: 'flex-end', paddingRight: 8,
                                                }}>
                                                    {f.count > 0 && <span style={{ fontSize: 'var(--text-xs)', fontWeight: 700, color: 'white' }}>{f.count}</span>}
                                                </div>
                                            </div>
                                            <span style={{ width: 30, textAlign: 'right', fontSize: 'var(--text-sm)', fontWeight: 700 }}>{f.count}</span>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </StaggeredItem>
                    )}

                    {/* PER AREA */}
                    {activeTab === 'area' && (
                        <div className="space-y-2">
                            {areaStats.length === 0 ? (
                                <div style={{ padding: '3rem', textAlign: 'center', color: 'var(--color-muted)', fontSize: 'var(--text-sm)' }}>
                                    Belum ada data area.
                                </div>
                            ) : (
                                areaStats.map((area, i) => (
                                    <StaggeredItem key={area.area} index={i}>
                                        <div className="card card-hover-lift" style={{ padding: '1rem' }}>
                                            <div className="flex items-center justify-between">
                                                <div className="flex items-center gap-3">
                                                    <MapPin size={16} style={{ color: 'var(--color-muted)' }} />
                                                    <div>
                                                        <h4 style={{ fontWeight: 700, fontSize: 'var(--text-sm)' }}>{area.area}</h4>
                                                        <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>{area.leads} leads · {area.psb} PSB</p>
                                                    </div>
                                                </div>
                                                <div style={{ textAlign: 'right' }}>
                                                    <div style={{ fontSize: 'var(--text-lg)', fontWeight: 700 }}>
                                                        {area.leads > 0 ? Math.round((area.psb / area.leads) * 100) : 0}%
                                                    </div>
                                                    <div style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>Conv. Rate</div>
                                                </div>
                                            </div>
                                        </div>
                                    </StaggeredItem>
                                ))
                            )}
                        </div>
                    )}
                </>
            )}
        </AnimatedPage>
    )
}
