'use client'

import { useState, useEffect } from 'react'
import {
    User, Target, Bell, Database, Link2, Settings,
    Zap, Save, Plus, X, Edit3, Check, ChevronDown
} from 'lucide-react'

// ── Tab Config ────────────────────────────────────────────
const settingsTabs = [
    { key: 'profil', label: 'Profil', icon: User },
    { key: 'target', label: 'Target & KPI', icon: Target },
    { key: 'notifikasi', label: 'Notifikasi', icon: Bell },
    { key: 'masterdata', label: 'Master Data', icon: Database },
    { key: 'integrasi', label: 'Integrasi', icon: Link2 },
    { key: 'sistem', label: 'Sistem', icon: Settings },
]

function formatRole(role: string) {
    return role.replace(/_/g, ' ').replace(/\b\w/g, c => c.toUpperCase())
}

// ── Toggle Component ──────────────────────────────────────
function Toggle({ enabled, onChange }: { enabled: boolean; onChange: () => void }) {
    return (
        <button
            type="button"
            onClick={onChange}
            className="relative"
            style={{
                width: 44, height: 24, borderRadius: 12,
                background: enabled ? 'var(--color-success)' : '#CBD5E1',
                transition: 'background 0.2s ease', border: 'none', cursor: 'pointer',
            }}
        >
            <span style={{
                position: 'absolute', top: 2, left: enabled ? 22 : 2,
                width: 20, height: 20, borderRadius: '50%', background: 'white',
                transition: 'left 0.2s ease', boxShadow: '0 1px 3px rgba(0,0,0,0.15)',
            }} />
        </button>
    )
}

// ── Main Component ────────────────────────────────────────
export default function SettingsPage() {
    const [activeTab, setActiveTab] = useState('profil')

    // Load user from localStorage
    const [userData, setUserData] = useState({ nama: '', email: '', role: '', noWa: '' })
    useEffect(() => {
        try {
            const stored = localStorage.getItem('jabnet_user')
            if (stored) {
                const u = JSON.parse(stored)
                setUserData({
                    nama: u.nama || 'User',
                    email: u.email || '',
                    role: u.role || '',
                    noWa: u.noWa || '',
                })
            }
        } catch { }
    }, [])

    const initials = userData.nama.split(' ').map(w => w[0]).join('').slice(0, 2).toUpperCase() || 'U'

    // Master Data state
    const [pakets, setPakets] = useState([
        { nama: 'To The Moon', speed: '20 Mbps', harga: 189000 },
        { nama: 'Galaksi', speed: '30 Mbps', harga: 219000 },
        { nama: 'Nebula', speed: '50 Mbps', harga: 279000 },
        { nama: 'Supernova', speed: '100 Mbps', harga: 399000 },
    ])
    const [lostReasons, setLostReasons] = useState([
        'Kompetitor', 'Harga', 'Tidak Berminat', 'Coverage Belum Ada', 'Pindah Rumah', 'Tidak Bisa Dihubungi',
    ])
    const [editingPaketIdx, setEditingPaketIdx] = useState<number | null>(null)
    const [editPaketForm, setEditPaketForm] = useState({ nama: '', speed: '', harga: 0 })
    const [showAddPaket, setShowAddPaket] = useState(false)
    const [newPaket, setNewPaket] = useState({ nama: '', speed: '', harga: 0 })
    const [newReason, setNewReason] = useState('')
    const [showAddReason, setShowAddReason] = useState(false)

    // Load master data from localStorage
    useEffect(() => {
        try {
            const stored = localStorage.getItem('jabnet-master-data')
            if (stored) {
                const d = JSON.parse(stored)
                if (d.pakets) setPakets(d.pakets)
                if (d.lostReasons) setLostReasons(d.lostReasons)
            }
        } catch { }
    }, [])

    // Save master data to localStorage
    const saveMasterData = (p: typeof pakets, r: typeof lostReasons) => {
        localStorage.setItem('jabnet-master-data', JSON.stringify({ pakets: p, lostReasons: r }))
    }

    // Notification toggles
    const [notifs, setNotifs] = useState({
        followUp: true,
        hotLead: true,
        dailyReport: true,
        closingAlert: true,
        visitReminder: false,
        whatsapp: true,
    })

    return (
        <div className="max-w-3xl mx-auto">
            <h1 style={{ fontSize: 'var(--text-2xl)', fontWeight: 700, marginBottom: '1.5rem' }}>Pengaturan</h1>

            {/* Tabs */}
            <div className="tab-bar" style={{ marginBottom: '1.5rem' }}>
                {settingsTabs.map(t => {
                    const Icon = t.icon
                    return (
                        <button
                            key={t.key}
                            className={`tab-item ${activeTab === t.key ? 'active' : ''}`}
                            onClick={() => setActiveTab(t.key)}
                        >
                            <span className="flex items-center gap-1.5">
                                <Icon size={14} />
                                <span className="hidden sm:inline">{t.label}</span>
                            </span>
                        </button>
                    )
                })}
            </div>

            {/* ── TAB: PROFIL ──────────────────────────── */}
            {activeTab === 'profil' && (
                <div className="space-y-4">
                    <div className="card" style={{ padding: '1.5rem' }}>
                        <div className="flex items-center gap-4 mb-6">
                            <div className="user-avatar" style={{ width: 64, height: 64, fontSize: 'var(--text-xl)', background: 'var(--color-subtle)' }}>
                                {initials}
                            </div>
                            <div>
                                <h2 style={{ fontSize: 'var(--text-xl)', fontWeight: 700 }}>{userData.nama || 'User'}</h2>
                                <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>{formatRole(userData.role)}</p>
                            </div>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label className="form-label">Nama Lengkap</label>
                                <input className="form-input" defaultValue={userData.nama} readOnly style={{ background: 'var(--color-subtle)' }} />
                            </div>
                            <div>
                                <label className="form-label">Email</label>
                                <input className="form-input" defaultValue={userData.email} readOnly type="email" style={{ background: 'var(--color-subtle)' }} />
                            </div>
                            <div>
                                <label className="form-label">No. WhatsApp</label>
                                <input className="form-input" defaultValue={userData.noWa} readOnly style={{ background: 'var(--color-subtle)' }} />
                            </div>
                            <div>
                                <label className="form-label">Role</label>
                                <input className="form-input" defaultValue={formatRole(userData.role)} disabled style={{ background: 'var(--color-subtle)', color: 'var(--color-muted)' }} />
                            </div>
                        </div>
                    </div>
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <h3 style={{ fontWeight: 700, marginBottom: '0.75rem' }}>Ubah Password</h3>
                        <p style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>Fitur ubah password belum tersedia di versi ini. Hubungi administrator untuk reset password.</p>
                    </div>
                </div>
            )}

            {/* ── TAB: TARGET & KPI ────────────────────── */}
            {activeTab === 'target' && (
                <div className="space-y-4">
                    {userData.role !== 'super_admin' && (
                        <div style={{ padding: '0.75rem 1rem', background: '#FFFBEB', border: '1px solid #FDE68A', borderRadius: 'var(--radius-md)', fontSize: 'var(--text-sm)', color: '#92400E' }}>
                            ⚠️ Hanya Super Admin yang dapat mengubah target & KPI.
                        </div>
                    )}
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <h3 style={{ fontWeight: 700, marginBottom: '1rem' }}>Target Tim (Bulan Ini)</h3>
                        <div className="space-y-4">
                            {[
                                { label: 'Target PSB Total', value: '50', unit: 'PSB/bulan' },
                                { label: 'Target Visit Harian (per sales)', value: '10', unit: 'visit/hari' },
                                { label: 'Target Leads Baru', value: '167', unit: 'leads/bulan' },
                                { label: 'Min. Closing Rate', value: '30', unit: '%' },
                            ].map((t, i) => (
                                <div key={i} className="flex items-center justify-between" style={{ padding: '0.75rem', background: 'var(--color-subtle)', borderRadius: 'var(--radius-md)' }}>
                                    <div>
                                        <p style={{ fontWeight: 600, fontSize: 'var(--text-sm)' }}>{t.label}</p>
                                        <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>{t.unit}</p>
                                    </div>
                                    {userData.role === 'super_admin' ? (
                                        <input className="form-input" defaultValue={t.value} style={{ width: 80, textAlign: 'center', height: 36 }} />
                                    ) : (
                                        <span style={{ fontWeight: 700, fontSize: 'var(--text-lg)', color: 'var(--color-text)' }}>{t.value}</span>
                                    )}
                                </div>
                            ))}
                        </div>
                        {userData.role === 'super_admin' && (
                            <button className="btn-accent mt-4">
                                <Save size={14} /> Simpan Target
                            </button>
                        )}
                    </div>
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <h3 style={{ fontWeight: 700, marginBottom: '0.75rem' }}>Target Per Sales</h3>
                        <div className="space-y-2">
                            {[
                                { nama: 'Herdiansyah', psb: 18, visit: 10 },
                                { nama: 'Rizky Nurzaman', psb: 15, visit: 10 },
                                { nama: 'Yadi Nur A.', psb: 12, visit: 10 },
                                { nama: 'Raisya Amelia', psb: 8, visit: 10 },
                            ].map((s, i) => (
                                <div key={i} className="flex items-center gap-3 flex-wrap" style={{ padding: '0.5rem 0.75rem', borderBottom: '1px solid var(--color-subtle)' }}>
                                    <span style={{ fontWeight: 600, fontSize: 'var(--text-sm)', flex: 1 }}>{s.nama}</span>
                                    <div className="flex items-center gap-2">
                                        <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>PSB:</span>
                                        {userData.role === 'super_admin' ? (
                                            <input className="form-input" defaultValue={s.psb} style={{ width: 50, height: 28, fontSize: 'var(--text-xs)', textAlign: 'center', padding: '0 4px' }} />
                                        ) : (
                                            <span style={{ fontWeight: 700, fontSize: 'var(--text-sm)' }}>{s.psb}</span>
                                        )}
                                        <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>Visit:</span>
                                        {userData.role === 'super_admin' ? (
                                            <input className="form-input" defaultValue={s.visit} style={{ width: 50, height: 28, fontSize: 'var(--text-xs)', textAlign: 'center', padding: '0 4px' }} />
                                        ) : (
                                            <span style={{ fontWeight: 700, fontSize: 'var(--text-sm)' }}>{s.visit}</span>
                                        )}
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
            )}

            {/* ── TAB: NOTIFIKASI ──────────────────────── */}
            {activeTab === 'notifikasi' && (
                <div className="card" style={{ padding: '1.25rem' }}>
                    <h3 style={{ fontWeight: 700, marginBottom: '1rem' }}>Pengaturan Notifikasi</h3>
                    <div className="space-y-0">
                        {[
                            { key: 'followUp' as keyof typeof notifs, title: 'Follow-up Reminder', desc: 'Ingatkan jadwal follow-up via in-app + WA' },
                            { key: 'hotLead' as keyof typeof notifs, title: 'Hot Lead Alert', desc: 'Alert jika hot lead belum di-follow-up >24 jam' },
                            { key: 'dailyReport' as keyof typeof notifs, title: 'Laporan Harian Otomatis', desc: 'Ringkasan performa tim setiap jam 17:00' },
                            { key: 'closingAlert' as keyof typeof notifs, title: 'Closing Notification', desc: 'Alert setiap ada closing baru dari tim' },
                            { key: 'visitReminder' as keyof typeof notifs, title: 'Visit Reminder', desc: 'Pengingat jika visit harian di bawah target' },
                            { key: 'whatsapp' as keyof typeof notifs, title: 'WhatsApp Integration', desc: 'Kirim notifikasi juga via WhatsApp (n8n)' },
                        ].map((n) => (
                            <div key={n.key} className="flex items-center justify-between" style={{ padding: '0.875rem 0', borderBottom: '1px solid var(--color-subtle)' }}>
                                <div>
                                    <p style={{ fontWeight: 600, fontSize: 'var(--text-sm)' }}>{n.title}</p>
                                    <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginTop: 2 }}>{n.desc}</p>
                                </div>
                                <Toggle
                                    enabled={notifs[n.key]}
                                    onChange={() => setNotifs(prev => ({ ...prev, [n.key]: !prev[n.key] }))}
                                />
                            </div>
                        ))}
                    </div>
                </div>
            )}

            {/* ── TAB: MASTER DATA ─────────────────────── */}
            {activeTab === 'masterdata' && (
                <div className="space-y-4">
                    {/* Paket */}
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <div className="flex items-center justify-between mb-3">
                            <h3 style={{ fontWeight: 700 }}>Master Paket</h3>
                            <button className="btn-ghost" style={{ fontSize: 'var(--text-xs)', color: 'var(--color-accent-500)' }} onClick={() => { setShowAddPaket(true); setNewPaket({ nama: '', speed: '', harga: 0 }) }}>
                                <Plus size={12} /> Tambah
                            </button>
                        </div>

                        {/* Add Paket Form */}
                        {showAddPaket && (
                            <div style={{ padding: '0.75rem', background: 'var(--color-subtle)', borderRadius: 'var(--radius-md)', marginBottom: '0.75rem' }}>
                                <div className="grid grid-cols-1 sm:grid-cols-3 gap-2" style={{ marginBottom: '0.5rem' }}>
                                    <input className="form-input" placeholder="Nama paket" value={newPaket.nama} onChange={e => setNewPaket(p => ({ ...p, nama: e.target.value }))} style={{ height: 36, fontSize: 'var(--text-sm)' }} />
                                    <input className="form-input" placeholder="Speed (mis: 20 Mbps)" value={newPaket.speed} onChange={e => setNewPaket(p => ({ ...p, speed: e.target.value }))} style={{ height: 36, fontSize: 'var(--text-sm)' }} />
                                    <input className="form-input" placeholder="Harga" type="number" value={newPaket.harga || ''} onChange={e => setNewPaket(p => ({ ...p, harga: parseInt(e.target.value) || 0 }))} style={{ height: 36, fontSize: 'var(--text-sm)' }} />
                                </div>
                                <div className="flex gap-2">
                                    <button className="btn-accent" style={{ fontSize: 'var(--text-xs)', padding: '4px 12px' }} onClick={() => {
                                        if (!newPaket.nama.trim()) return
                                        const updated = [...pakets, newPaket]
                                        setPakets(updated)
                                        saveMasterData(updated, lostReasons)
                                        setShowAddPaket(false)
                                    }}><Check size={12} /> Simpan</button>
                                    <button className="btn-ghost" style={{ fontSize: 'var(--text-xs)', padding: '4px 12px' }} onClick={() => setShowAddPaket(false)}><X size={12} /> Batal</button>
                                </div>
                            </div>
                        )}

                        <div className="space-y-0">
                            {pakets.map((p, i) => (
                                <div key={i} style={{ padding: '0.625rem 0', borderBottom: '1px solid var(--color-subtle)' }}>
                                    {editingPaketIdx === i ? (
                                        /* Editing Mode */
                                        <div>
                                            <div className="grid grid-cols-1 sm:grid-cols-3 gap-2" style={{ marginBottom: '0.5rem' }}>
                                                <input className="form-input" value={editPaketForm.nama} onChange={e => setEditPaketForm(f => ({ ...f, nama: e.target.value }))} style={{ height: 36, fontSize: 'var(--text-sm)' }} />
                                                <input className="form-input" value={editPaketForm.speed} onChange={e => setEditPaketForm(f => ({ ...f, speed: e.target.value }))} style={{ height: 36, fontSize: 'var(--text-sm)' }} />
                                                <input className="form-input" type="number" value={editPaketForm.harga || ''} onChange={e => setEditPaketForm(f => ({ ...f, harga: parseInt(e.target.value) || 0 }))} style={{ height: 36, fontSize: 'var(--text-sm)' }} />
                                            </div>
                                            <div className="flex gap-2">
                                                <button className="btn-accent" style={{ fontSize: 'var(--text-xs)', padding: '4px 12px' }} onClick={() => {
                                                    const updated = pakets.map((pk, idx) => idx === i ? editPaketForm : pk)
                                                    setPakets(updated)
                                                    saveMasterData(updated, lostReasons)
                                                    setEditingPaketIdx(null)
                                                }}><Check size={12} /> Simpan</button>
                                                <button className="btn-ghost" style={{ fontSize: 'var(--text-xs)', padding: '4px 12px' }} onClick={() => setEditingPaketIdx(null)}><X size={12} /> Batal</button>
                                                <button style={{ fontSize: 'var(--text-xs)', padding: '4px 12px', background: 'none', border: 'none', color: 'var(--color-danger)', cursor: 'pointer', fontWeight: 600 }} onClick={() => {
                                                    const updated = pakets.filter((_, idx) => idx !== i)
                                                    setPakets(updated)
                                                    saveMasterData(updated, lostReasons)
                                                    setEditingPaketIdx(null)
                                                }}>Hapus</button>
                                            </div>
                                        </div>
                                    ) : (
                                        /* Display Mode */
                                        <div className="flex items-center justify-between">
                                            <div>
                                                <span style={{ fontWeight: 600, fontSize: 'var(--text-sm)' }}>{p.nama}</span>
                                                <span style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginLeft: 8 }}>{p.speed}</span>
                                            </div>
                                            <div className="flex items-center gap-2">
                                                <span style={{ fontWeight: 700, fontSize: 'var(--text-sm)' }}>Rp {p.harga.toLocaleString('id-ID')}</span>
                                                <button className="btn-ghost" style={{ padding: 4 }} onClick={() => {
                                                    setEditingPaketIdx(i)
                                                    setEditPaketForm({ ...p })
                                                }}><Edit3 size={12} /></button>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            ))}
                        </div>
                    </div>

                    {/* Loss Reasons */}
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <div className="flex items-center justify-between mb-3">
                            <h3 style={{ fontWeight: 700 }}>Alasan Lost</h3>
                            <button className="btn-ghost" style={{ fontSize: 'var(--text-xs)', color: 'var(--color-accent-500)' }} onClick={() => setShowAddReason(true)}>
                                <Plus size={12} /> Tambah
                            </button>
                        </div>

                        {showAddReason && (
                            <div className="flex gap-2" style={{ marginBottom: '0.75rem' }}>
                                <input className="form-input" placeholder="Alasan baru..." value={newReason} onChange={e => setNewReason(e.target.value)} onKeyDown={e => {
                                    if (e.key === 'Enter' && newReason.trim()) {
                                        const updated = [...lostReasons, newReason.trim()]
                                        setLostReasons(updated)
                                        saveMasterData(pakets, updated)
                                        setNewReason('')
                                        setShowAddReason(false)
                                    }
                                }} style={{ height: 36, fontSize: 'var(--text-sm)', flex: 1 }} autoFocus />
                                <button className="btn-accent" style={{ fontSize: 'var(--text-xs)', padding: '4px 12px', height: 36 }} onClick={() => {
                                    if (!newReason.trim()) return
                                    const updated = [...lostReasons, newReason.trim()]
                                    setLostReasons(updated)
                                    saveMasterData(pakets, updated)
                                    setNewReason('')
                                    setShowAddReason(false)
                                }}><Check size={12} /></button>
                                <button className="btn-ghost" style={{ fontSize: 'var(--text-xs)', padding: '4px 12px', height: 36 }} onClick={() => { setShowAddReason(false); setNewReason('') }}><X size={12} /></button>
                            </div>
                        )}

                        <div className="flex flex-wrap gap-2">
                            {lostReasons.map((r, i) => (
                                <span key={i} className="stage-badge" style={{ background: 'var(--color-subtle)', color: 'var(--color-text)', padding: '4px 12px', display: 'inline-flex', alignItems: 'center', gap: 6, cursor: 'default' }}>
                                    {r}
                                    <button onClick={() => {
                                        const updated = lostReasons.filter((_, idx) => idx !== i)
                                        setLostReasons(updated)
                                        saveMasterData(pakets, updated)
                                    }} style={{ background: 'none', border: 'none', cursor: 'pointer', color: 'var(--color-muted)', padding: 0, lineHeight: 1, display: 'flex' }}>
                                        <X size={10} />
                                    </button>
                                </span>
                            ))}
                        </div>
                    </div>
                </div>
            )}

            {/* ── TAB: INTEGRASI ────────────────────────── */}
            {activeTab === 'integrasi' && (
                <div className="space-y-4">
                    {[
                        { name: 'n8n Automation', status: 'Connected', statusColor: 'var(--color-success)', desc: 'Workflow automation untuk follow-up WA', url: 'https://n8n.jabnet.id' },
                        { name: 'WhatsApp Business API', status: 'Connected', statusColor: 'var(--color-success)', desc: 'Kirim pesan otomatis via WA', url: 'Melalui n8n' },
                        { name: 'Google Maps API', status: 'Not Connected', statusColor: 'var(--color-muted)', desc: 'Untuk GPS capture dan mapping area', url: '' },
                    ].map((int, i) => (
                        <div key={i} className="card" style={{ padding: '1.25rem' }}>
                            <div className="flex items-center justify-between">
                                <div className="flex items-center gap-3">
                                    <div style={{ width: 40, height: 40, borderRadius: 'var(--radius-md)', background: 'var(--color-subtle)', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
                                        <Link2 size={18} style={{ color: 'var(--color-muted)' }} />
                                    </div>
                                    <div>
                                        <p style={{ fontWeight: 700 }}>{int.name}</p>
                                        <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>{int.desc}</p>
                                    </div>
                                </div>
                                <span className="stage-badge" style={{ background: int.status === 'Connected' ? '#ECFDF5' : '#F1F5F9', color: int.statusColor }}>
                                    {int.status}
                                </span>
                            </div>
                            {int.url && (
                                <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)', marginTop: 8, paddingLeft: 52 }}>{int.url}</p>
                            )}
                        </div>
                    ))}
                </div>
            )}

            {/* ── TAB: SISTEM ──────────────────────────── */}
            {activeTab === 'sistem' && (
                <div className="space-y-4">
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <h3 style={{ fontWeight: 700, marginBottom: '1rem' }}>Informasi Sistem</h3>
                        <div className="space-y-0">
                            {[
                                { label: 'Versi Aplikasi', value: 'v2.0.0' },
                                { label: 'Framework', value: 'Next.js 16 + Prisma 5' },
                                { label: 'Database', value: 'SQLite (local)' },
                                { label: 'Build', value: 'Feb 12, 2026' },
                                { label: 'Environment', value: 'Development' },
                            ].map((s, i) => (
                                <div key={i} className="flex items-center justify-between" style={{ padding: '0.625rem 0', borderBottom: '1px solid var(--color-subtle)' }}>
                                    <span style={{ fontSize: 'var(--text-sm)', color: 'var(--color-muted)' }}>{s.label}</span>
                                    <span style={{ fontSize: 'var(--text-sm)', fontWeight: 600 }}>{s.value}</span>
                                </div>
                            ))}
                        </div>
                    </div>
                    <div className="card" style={{ padding: '1.25rem' }}>
                        <h3 style={{ fontWeight: 700, marginBottom: '0.75rem' }}>Developer</h3>
                        <div className="flex items-center gap-3">
                            <div className="sidebar-logo-icon" style={{ width: 36, height: 36 }}>
                                <Zap size={16} className="text-white" />
                            </div>
                            <div>
                                <p style={{ fontWeight: 700 }}>JABNET Sales CRM</p>
                                <p style={{ fontSize: 'var(--text-xs)', color: 'var(--color-muted)' }}>
                                    © 2026 PT Jujur Amanah Barokah · Powered by Antigravity
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    )
}
