import { prisma } from '@/lib/prisma'
import { NextResponse } from 'next/server'
import bcrypt from 'bcryptjs'

// POST /api/auth/login - Authenticate user
export async function POST(request: Request) {
    try {
        const body = await request.json()
        const { email, password } = body

        if (!email || !password) {
            return NextResponse.json({ error: 'Email dan password wajib diisi.' }, { status: 400 })
        }

        const user = await prisma.user.findUnique({
            where: { email },
            select: {
                id: true,
                nama: true,
                email: true,
                password: true,
                role: true,
                noWa: true,
                isActive: true,
            },
        })

        if (!user) {
            return NextResponse.json({ error: 'Kredensial tidak valid.' }, { status: 401 })
        }

        if (!user.isActive) {
            return NextResponse.json({ error: 'Akun Anda tidak aktif. Hubungi administrator.' }, { status: 403 })
        }

        const isValid = await bcrypt.compare(password, user.password)
        if (!isValid) {
            return NextResponse.json({ error: 'Kredensial tidak valid.' }, { status: 401 })
        }

        // Return user data (without password)
        const { password: _, ...userData } = user
        return NextResponse.json(userData)
    } catch (error) {
        console.error('POST /api/auth/login error:', error)
        return NextResponse.json({ error: 'Terjadi kesalahan server.' }, { status: 500 })
    }
}
