import { prisma } from '@/lib/prisma'
import { NextResponse } from 'next/server'
import bcrypt from 'bcryptjs'

// GET /api/users - List all users
export async function GET() {
    try {
        const users = await prisma.user.findMany({
            orderBy: { createdAt: 'asc' },
            select: {
                id: true,
                nama: true,
                email: true,
                role: true,
                noWa: true,
                isActive: true,
                targetPsbMonthly: true,
                targetVisitDaily: true,
                createdAt: true,
            },
        })
        return NextResponse.json(users)
    } catch (error) {
        console.error('GET /api/users error:', error)
        return NextResponse.json({ error: 'Failed to fetch users' }, { status: 500 })
    }
}

// POST /api/users - Create a new user
export async function POST(request: Request) {
    try {
        const body = await request.json()
        const { nama, email, password, role, noWa, targetPsbMonthly, targetVisitDaily } = body

        if (!nama || !email || !password || !noWa) {
            return NextResponse.json({ error: 'Nama, email, password, dan nomor WA wajib diisi.' }, { status: 400 })
        }

        // Check duplicate email
        const existing = await prisma.user.findUnique({ where: { email } })
        if (existing) {
            return NextResponse.json({ error: 'Email sudah terdaftar.' }, { status: 409 })
        }

        const hashedPassword = await bcrypt.hash(password, 10)

        const user = await prisma.user.create({
            data: {
                nama,
                email,
                password: hashedPassword,
                role: role || 'sales_junior',
                noWa,
                targetPsbMonthly: targetPsbMonthly || 15,
                targetVisitDaily: targetVisitDaily || 10,
            },
            select: {
                id: true,
                nama: true,
                email: true,
                role: true,
                noWa: true,
                isActive: true,
                targetPsbMonthly: true,
                targetVisitDaily: true,
                createdAt: true,
            },
        })

        return NextResponse.json(user, { status: 201 })
    } catch (error) {
        console.error('POST /api/users error:', error)
        return NextResponse.json({ error: 'Failed to create user' }, { status: 500 })
    }
}
