'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import { Zap, Loader2, ArrowRight, Eye, EyeOff, ShieldAlert } from 'lucide-react'

export default function LoginPage() {
    const router = useRouter()
    const [email, setEmail] = useState('')
    const [password, setPassword] = useState('')
    const [showPassword, setShowPassword] = useState(false)
    const [loading, setLoading] = useState(false)
    const [error, setError] = useState('')
    const [mounted, setMounted] = useState(false)

    useEffect(() => {
        setMounted(true)
    }, [])

    const handleLogin = async (e: React.FormEvent) => {
        e.preventDefault()
        setLoading(true)
        setError('')

        try {
            const res = await fetch('/api/auth/login', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email, password }),
            })

            const data = await res.json()

            if (res.ok) {
                localStorage.setItem('jabnet_user', JSON.stringify({
                    id: data.id,
                    email: data.email,
                    role: data.role,
                    nama: data.nama,
                    noWa: data.noWa,
                }))
                router.push('/dashboard')
            } else {
                setError(data.error || 'Kredensial tidak valid. Silakan periksa email dan password Anda.')
                setLoading(false)
            }
        } catch {
            setError('Tidak dapat terhubung ke server. Periksa koneksi Anda.')
            setLoading(false)
        }
    }

    if (!mounted) return null

    return (
        <div style={{ minHeight: '100dvh', width: '100%', display: 'flex', background: '#fff', fontFamily: 'var(--font-sans, system-ui, sans-serif)', color: '#0f172a' }}>
            {/* LEFT SIDE: Brand Identity (Desktop Only) */}
            <div className="hidden lg:flex" style={{ width: '45%', position: 'relative', background: '#0f172a', color: 'white', flexDirection: 'column', justifyContent: 'space-between', overflow: 'hidden' }}>
                {/* Background effects */}
                <div style={{ position: 'absolute', inset: 0, zIndex: 0, opacity: 0.03 }} />
                <div style={{ position: 'absolute', top: 0, right: 0, width: 500, height: 500, background: 'radial-gradient(circle, rgba(249,115,22,0.2), rgba(147,51,234,0.1))', filter: 'blur(120px)', borderRadius: '100%', transform: 'translateX(33%) translateY(-33%)' }} />
                <div style={{ position: 'absolute', bottom: 0, left: 0, width: 400, height: 400, background: 'radial-gradient(circle, rgba(37,99,235,0.1), rgba(45,212,191,0.1))', filter: 'blur(100px)', borderRadius: '100%', transform: 'translateX(-33%) translateY(33%)' }} />

                <div style={{ position: 'relative', zIndex: 10, padding: 48, height: '100%', display: 'flex', flexDirection: 'column', justifyContent: 'space-between' }}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: 12 }}>
                        <div style={{ width: 40, height: 40, borderRadius: 8, background: '#EA580C', display: 'flex', alignItems: 'center', justifyContent: 'center', boxShadow: '0 4px 12px rgba(234,88,12,0.2)' }}>
                            <Zap size={20} color="white" fill="white" />
                        </div>
                        <span style={{ fontSize: 20, fontWeight: 700, letterSpacing: '-0.01em', opacity: 0.9 }}>JABNET CRM</span>
                    </div>

                    <div style={{ maxWidth: 400 }}>
                        <h1 style={{ fontSize: 36, fontWeight: 700, lineHeight: 1.2, marginBottom: 24, letterSpacing: '-0.02em' }}>
                            Smart solution for modern sales teams.
                        </h1>
                        <p style={{ fontSize: 18, color: '#94a3b8', lineHeight: 1.7, fontWeight: 300 }}>
                            Manage leads, track performance, and close deals faster with our integrated CRM platform.
                        </p>
                    </div>

                    <div style={{ borderTop: '1px solid rgba(255,255,255,0.1)', paddingTop: 32 }}>
                        <p style={{ fontSize: 13, color: '#64748b', fontWeight: 500, letterSpacing: '0.05em', textTransform: 'uppercase' }}>
                            Enterprise Solution © 2026 PT Jujur Amanah Barokah
                        </p>
                    </div>
                </div>
            </div>

            {/* RIGHT SIDE: Login Form */}
            <div style={{
                width: '100%', display: 'flex', alignItems: 'center', justifyContent: 'center',
                padding: '24px 20px', position: 'relative', background: '#fff',
                paddingTop: 'max(24px, env(safe-area-inset-top, 24px))',
                paddingBottom: 'max(24px, env(safe-area-inset-bottom, 24px))',
            }}>
                <div style={{ width: '100%', maxWidth: 440 }}>

                    {/* Mobile Logo */}
                    <div className="lg:hidden" style={{ display: 'flex', alignItems: 'center', marginBottom: 32 }}>
                        <div style={{ width: 44, height: 44, borderRadius: 12, background: '#EA580C', display: 'flex', alignItems: 'center', justifyContent: 'center', boxShadow: '0 4px 12px rgba(234,88,12,0.2)', marginRight: 12 }}>
                            <Zap size={22} color="white" fill="white" />
                        </div>
                        <span style={{ fontSize: 22, fontWeight: 700, letterSpacing: '-0.01em', color: '#0f172a' }}>JABNET CRM</span>
                    </div>

                    <div style={{ marginBottom: 24 }}>
                        <h2 style={{ fontSize: 'clamp(24px, 6vw, 36px)', fontWeight: 700, color: '#0f172a', letterSpacing: '-0.02em', marginBottom: 6 }}>Welcome back</h2>
                        <p style={{ color: '#64748b', fontSize: 'clamp(14px, 3.5vw, 18px)' }}>Please enter your details to sign in.</p>
                    </div>

                    {error && (
                        <div style={{ padding: '12px 14px', background: '#FEF2F2', border: '1px solid #FEE2E2', borderRadius: 12, display: 'flex', alignItems: 'flex-start', gap: 10, marginBottom: 20 }}>
                            <ShieldAlert size={18} color="#EF4444" style={{ flexShrink: 0, marginTop: 1 }} />
                            <p style={{ fontSize: 13, color: '#DC2626', fontWeight: 500 }}>{error}</p>
                        </div>
                    )}

                    <form onSubmit={handleLogin}>
                        <div style={{ marginBottom: 16 }}>
                            <label style={{ fontSize: 13, fontWeight: 600, color: '#0f172a', display: 'block', marginBottom: 6 }}>Email Address</label>
                            <input
                                type="email"
                                style={{
                                    width: '100%', padding: '12px 16px', background: '#fff',
                                    border: '1px solid #e2e8f0', borderRadius: 12,
                                    fontSize: 15, color: '#0f172a', fontWeight: 500,
                                    outline: 'none', transition: 'border-color 0.2s, box-shadow 0.2s',
                                    minHeight: 48, boxSizing: 'border-box',
                                }}
                                placeholder="name@company.com"
                                value={email}
                                onChange={(e) => setEmail(e.target.value)}
                                onFocus={(e) => { e.target.style.borderColor = '#EA580C'; e.target.style.boxShadow = '0 0 0 3px rgba(234,88,12,0.1)' }}
                                onBlur={(e) => { e.target.style.borderColor = '#e2e8f0'; e.target.style.boxShadow = 'none' }}
                                required
                                autoFocus
                                autoComplete="email"
                            />
                        </div>

                        <div style={{ marginBottom: 8 }}>
                            <label style={{ fontSize: 13, fontWeight: 600, color: '#0f172a', display: 'block', marginBottom: 6 }}>Password</label>
                            <div style={{ position: 'relative' }}>
                                <input
                                    type={showPassword ? 'text' : 'password'}
                                    style={{
                                        width: '100%', padding: '12px 48px 12px 16px', background: '#fff',
                                        border: '1px solid #e2e8f0', borderRadius: 12,
                                        fontSize: 15, color: '#0f172a', fontWeight: 500,
                                        outline: 'none', transition: 'border-color 0.2s, box-shadow 0.2s',
                                        minHeight: 48, boxSizing: 'border-box',
                                    }}
                                    placeholder="Enter your password"
                                    value={password}
                                    onChange={(e) => setPassword(e.target.value)}
                                    onFocus={(e) => { e.target.style.borderColor = '#EA580C'; e.target.style.boxShadow = '0 0 0 3px rgba(234,88,12,0.1)' }}
                                    onBlur={(e) => { e.target.style.borderColor = '#e2e8f0'; e.target.style.boxShadow = 'none' }}
                                    required
                                    autoComplete="current-password"
                                />
                                <button
                                    type="button"
                                    onClick={() => setShowPassword(!showPassword)}
                                    style={{
                                        position: 'absolute', right: 4, top: '50%', transform: 'translateY(-50%)',
                                        background: 'none', border: 'none', cursor: 'pointer', color: '#94a3b8',
                                        padding: 8, display: 'flex', alignItems: 'center', justifyContent: 'center',
                                        minWidth: 40, minHeight: 40,
                                    }}
                                >
                                    {showPassword ? <EyeOff size={18} /> : <Eye size={18} />}
                                </button>
                            </div>
                            <div style={{ display: 'flex', justifyContent: 'flex-end', paddingTop: 6 }}>
                                <a href="#" style={{ fontSize: 13, fontWeight: 500, color: '#EA580C', textDecoration: 'none' }}>Forgot password?</a>
                            </div>
                        </div>

                        <div style={{ paddingTop: 8 }}>
                            <button
                                type="submit"
                                disabled={loading}
                                style={{
                                    width: '100%', padding: '14px 20px', background: loading ? '#334155' : '#0f172a',
                                    color: '#fff', fontSize: 16, fontWeight: 700, borderRadius: 12,
                                    border: 'none', cursor: loading ? 'not-allowed' : 'pointer',
                                    boxShadow: '0 4px 12px rgba(15,23,42,0.1)',
                                    transition: 'all 0.2s', display: 'flex', alignItems: 'center', justifyContent: 'center', gap: 10,
                                    minHeight: 52, opacity: loading ? 0.7 : 1,
                                }}
                            >
                                {loading ? (
                                    <>
                                        <Loader2 size={20} className="animate-spin" />
                                        <span>Signing in...</span>
                                    </>
                                ) : (
                                    <>
                                        <span>Sign in</span>
                                        <ArrowRight size={20} />
                                    </>
                                )}
                            </button>
                        </div>
                    </form>

                    {/* Footer */}
                    <div style={{ paddingTop: 20, textAlign: 'center', fontSize: 13, color: '#64748b', borderTop: '1px solid #f1f5f9', marginTop: 24 }}>
                        Don&apos;t have an account? <span style={{ color: '#0f172a', fontWeight: 700, cursor: 'pointer' }}>Contact Administrator</span>
                    </div>
                </div>
            </div>
        </div>
    )
}
