'use client'

import { usePathname } from 'next/navigation'
import Link from 'next/link'
import { useRouter } from 'next/navigation'
import {
    LayoutDashboard,
    Users,
    UserPlus,
    BarChart3,
    MapPin,
    Settings,
    LogOut,
    Menu,
    X,
    Zap,
    GitBranch,
    PanelLeftClose,
    PanelLeftOpen,
    ShieldCheck,
} from 'lucide-react'
import { useState, useEffect, useCallback, useRef } from 'react'

const navItems = [
    { href: '/dashboard', label: 'Dashboard', icon: LayoutDashboard, section: 'main' },
    { href: '/leads', label: 'My Leads', icon: Users, section: 'main' },
    { href: '/leads/new', label: 'Input Lead', icon: UserPlus, section: 'main' },
    { href: '/pipeline', label: 'Pipeline', icon: GitBranch, section: 'main' },
    { href: '/areas', label: 'Area', icon: MapPin, section: 'main' },
    { href: '/reports', label: 'Laporan', icon: BarChart3, section: 'other' },
    { href: '/admin/users', label: 'Admin', icon: ShieldCheck, section: 'other' },
    { href: '/settings', label: 'Pengaturan', icon: Settings, section: 'other' },
]

const mobileNavItems = [
    { href: '/dashboard', label: 'Home', icon: LayoutDashboard },
    { href: '/leads', label: 'Leads', icon: Users },
    { href: '/leads/new', label: 'Input', icon: UserPlus },
    { href: '/pipeline', label: 'Pipeline', icon: GitBranch },
    { href: '/settings', label: 'Lainnya', icon: Settings },
]

export default function Sidebar() {
    const pathname = usePathname()
    const router = useRouter()
    const [mobileOpen, setMobileOpen] = useState(false)
    const [collapsed, setCollapsed] = useState(false)
    const [userName, setUserName] = useState('User')
    const [userRole, setUserRole] = useState('')
    const sidebarRef = useRef<HTMLElement>(null)
    const touchStartX = useRef(0)
    const touchCurrentX = useRef(0)

    const handleLogout = () => {
        localStorage.removeItem('jabnet_user')
        localStorage.removeItem('crm-user-name')
        router.push('/login')
    }

    const isAdmin = userRole === 'super_admin' || userRole === 'admin'

    // Persist collapsed state & load user info
    useEffect(() => {
        const stored = localStorage.getItem('sidebar-collapsed')
        if (stored === 'true') setCollapsed(true)

        // Load user from localStorage
        try {
            const userData = localStorage.getItem('jabnet_user')
            if (userData) {
                const user = JSON.parse(userData)
                setUserName(user.nama || 'User')
                setUserRole(user.role || '')
                localStorage.setItem('crm-user-name', user.nama || 'User')
            }
        } catch { }
    }, [])

    // Close sidebar on route change (mobile)
    useEffect(() => {
        setMobileOpen(false)
    }, [pathname])

    // Prevent body scroll when mobile sidebar is open
    useEffect(() => {
        if (mobileOpen) {
            document.body.style.overflow = 'hidden'
        } else {
            document.body.style.overflow = ''
        }
        return () => { document.body.style.overflow = '' }
    }, [mobileOpen])

    const toggleCollapsed = () => {
        const next = !collapsed
        setCollapsed(next)
        localStorage.setItem('sidebar-collapsed', String(next))
        // Dispatch custom event so layout can listen
        window.dispatchEvent(new CustomEvent('sidebar-toggle', { detail: { collapsed: next } }))
    }

    // Swipe-to-close gesture support
    const handleTouchStart = useCallback((e: React.TouchEvent) => {
        touchStartX.current = e.touches[0].clientX
        touchCurrentX.current = e.touches[0].clientX
    }, [])

    const handleTouchMove = useCallback((e: React.TouchEvent) => {
        touchCurrentX.current = e.touches[0].clientX
        const delta = touchStartX.current - touchCurrentX.current
        // If swiping left, translate sidebar
        if (delta > 0 && sidebarRef.current) {
            const translate = Math.min(delta, 240)
            sidebarRef.current.style.transform = `translateX(-${translate}px)`
            sidebarRef.current.style.transition = 'none'
        }
    }, [])

    const handleTouchEnd = useCallback(() => {
        const delta = touchStartX.current - touchCurrentX.current
        if (sidebarRef.current) {
            sidebarRef.current.style.transition = ''
            sidebarRef.current.style.transform = ''
        }
        // If swiped left more than 80px, close
        if (delta > 80) {
            setMobileOpen(false)
        }
    }, [])

    const mainItems = navItems.filter(i => i.section === 'main')
    const otherItems = navItems.filter(i => {
        if (i.section !== 'other') return false
        // Only show Admin menu for super_admin and admin roles
        if (i.href === '/admin/users' && !isAdmin) return false
        return true
    })

    return (
        <>
            {/* Mobile Header */}
            <header
                className="md:hidden fixed top-0 left-0 right-0 bg-white z-50 flex items-center justify-between px-4"
                style={{
                    borderBottom: '1px solid var(--color-border)',
                    height: 56,
                    paddingTop: 'env(safe-area-inset-top, 0px)',
                }}
            >
                <button
                    onClick={() => setMobileOpen(true)}
                    style={{
                        padding: 8,
                        background: 'transparent',
                        border: 'none',
                        cursor: 'pointer',
                        borderRadius: 'var(--radius-md)',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        minWidth: 44,
                        minHeight: 44,
                    }}
                    aria-label="Open navigation menu"
                >
                    <Menu size={22} style={{ color: 'var(--color-text)' }} />
                </button>
                <div className="flex items-center gap-2">
                    <div className="sidebar-logo-icon" style={{ width: 28, height: 28 }}>
                        <Zap size={14} className="text-white" />
                    </div>
                    <span style={{ fontSize: 'var(--text-sm)', fontWeight: 700, color: 'var(--color-primary-900)' }}>
                        JABNET CRM
                    </span>
                </div>
                <button
                    onClick={handleLogout}
                    style={{
                        padding: 8,
                        background: 'transparent',
                        border: 'none',
                        cursor: 'pointer',
                        borderRadius: 'var(--radius-md)',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        minWidth: 44,
                        minHeight: 44,
                    }}
                    aria-label="Logout"
                    title="Logout"
                >
                    <LogOut size={20} style={{ color: 'var(--color-danger, #EF4444)' }} />
                </button>
            </header>

            {/* Mobile Overlay (backdrop) */}
            <div
                className={`sidebar-overlay ${mobileOpen ? 'visible' : ''}`}
                onClick={() => setMobileOpen(false)}
                style={{ display: 'none' }}
            />
            {/* Show overlay only on mobile via inline media query */}
            <style>{`
                @media (max-width: 768px) {
                    .sidebar-overlay { display: block !important; }
                }
            `}</style>

            {/* Sidebar */}
            <aside
                ref={sidebarRef}
                className={`sidebar ${mobileOpen ? 'open' : ''} ${collapsed ? 'collapsed' : ''}`}
                role="navigation"
                aria-label="Main navigation"
                onTouchStart={handleTouchStart}
                onTouchMove={handleTouchMove}
                onTouchEnd={handleTouchEnd}
            >
                {/* Logo */}
                <div className="sidebar-logo">
                    <div className="sidebar-logo-icon">
                        <Zap size={18} className="text-white" />
                    </div>
                    <div className="sidebar-logo-text">
                        <span className="text-sm font-bold tracking-tight">JABNET</span>
                        <span style={{ fontSize: '0.6rem', color: 'rgba(255,255,255,0.4)', fontWeight: 600, letterSpacing: '0.1em' }}>SALES CRM</span>
                    </div>
                    {/* Mobile close */}
                    <button
                        className="md:hidden ml-auto"
                        onClick={() => setMobileOpen(false)}
                        aria-label="Close navigation menu"
                        style={{
                            padding: 8,
                            background: 'transparent',
                            border: 'none',
                            cursor: 'pointer',
                            color: 'rgba(255,255,255,0.6)',
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center',
                            minWidth: 44,
                            minHeight: 44,
                            borderRadius: 'var(--radius-md)',
                        }}
                    >
                        <X size={20} />
                    </button>
                </div>

                {/* Navigation */}
                <nav className="sidebar-nav">
                    <div className="sidebar-section-label">Menu</div>
                    {mainItems.map((item) => {
                        const Icon = item.icon
                        const isActive = pathname === item.href || (item.href !== '/leads/new' && pathname.startsWith(item.href + '/'))
                        return (
                            <Link
                                key={item.href}
                                href={item.href}
                                className={`sidebar-link ${isActive ? 'active' : ''}`}
                                onClick={() => setMobileOpen(false)}
                            >
                                <span className="sidebar-link-icon">
                                    <Icon size={18} />
                                </span>
                                <span className="sidebar-link-text">{item.label}</span>
                                {collapsed && (
                                    <span className="sidebar-tooltip">{item.label}</span>
                                )}
                            </Link>
                        )
                    })}

                    <div className="sidebar-section-label" style={{ marginTop: '0.5rem' }}>Lainnya</div>
                    {otherItems.map((item) => {
                        const Icon = item.icon
                        const isActive = pathname === item.href || pathname.startsWith(item.href + '/')
                        return (
                            <Link
                                key={item.href}
                                href={item.href}
                                className={`sidebar-link ${isActive ? 'active' : ''}`}
                                onClick={() => setMobileOpen(false)}
                            >
                                <span className="sidebar-link-icon">
                                    <Icon size={18} />
                                </span>
                                <span className="sidebar-link-text">{item.label}</span>
                                {collapsed && (
                                    <span className="sidebar-tooltip">{item.label}</span>
                                )}
                            </Link>
                        )
                    })}
                </nav>

                {/* Collapse Toggle */}
                <button
                    className="sidebar-collapse-btn"
                    onClick={toggleCollapsed}
                    title={collapsed ? 'Expand sidebar' : 'Collapse sidebar'}
                >
                    {collapsed ? <PanelLeftOpen size={18} /> : <PanelLeftClose size={18} />}
                </button>

                {/* User */}
                <div className="sidebar-user">
                    <div className="sidebar-avatar">
                        {userName.split(' ').map(w => w[0]).join('').slice(0, 2).toUpperCase()}
                        <span className="online-dot" />
                    </div>
                    <div className="sidebar-user-info">
                        <p className="text-sm font-semibold truncate text-white">{userName}</p>
                        <p style={{ fontSize: '0.6875rem', color: 'rgba(255,255,255,0.5)' }}>{userRole.replace(/_/g, ' ').replace(/\b\w/g, c => c.toUpperCase())}</p>
                    </div>
                    <button
                        onClick={handleLogout}
                        className="p-1.5 text-gray-400 hover:text-white transition-colors ml-auto"
                        style={{ flexShrink: 0, minWidth: 44, minHeight: 44, display: 'flex', alignItems: 'center', justifyContent: 'center', background: 'transparent', border: 'none', cursor: 'pointer' }}
                        aria-label="Logout"
                        title="Logout"
                    >
                        <LogOut size={16} />
                    </button>
                </div>
            </aside>

            {/* Mobile Bottom Nav */}
            <nav className="mobile-nav" role="navigation" aria-label="Mobile navigation">
                {mobileNavItems.map((item) => {
                    const Icon = item.icon
                    const isActive = pathname === item.href || pathname.startsWith(item.href + '/')
                    return (
                        <Link
                            key={item.href}
                            href={item.href}
                            className={`mobile-nav-item ${isActive ? 'active' : ''}`}
                        >
                            <Icon size={20} strokeWidth={isActive ? 2.5 : 2} />
                            <span>{item.label}</span>
                        </Link>
                    )
                })}
            </nav>
        </>
    )
}
