'use client'

import { createContext, useContext, useState, useCallback, useEffect } from 'react'
import { CheckCircle, AlertTriangle, Info, X } from 'lucide-react'

type ToastType = 'success' | 'error' | 'info' | 'warning'

interface Toast {
    id: string
    type: ToastType
    title: string
    message?: string
}

interface ToastContextType {
    addToast: (type: ToastType, title: string, message?: string) => void
}

const ToastContext = createContext<ToastContextType>({ addToast: () => { } })
export const useToast = () => useContext(ToastContext)

const iconMap = {
    success: { icon: CheckCircle, color: '#10B981', bg: '#ECFDF5' },
    error: { icon: AlertTriangle, color: '#EF4444', bg: '#FEF2F2' },
    warning: { icon: AlertTriangle, color: '#F59E0B', bg: '#FFFBEB' },
    info: { icon: Info, color: '#3B82F6', bg: '#EFF6FF' },
}

function ToastItem({ toast, onRemove }: { toast: Toast; onRemove: (id: string) => void }) {
    const [visible, setVisible] = useState(false)
    const config = iconMap[toast.type]
    const Icon = config.icon

    useEffect(() => {
        requestAnimationFrame(() => setVisible(true))
        const timer = setTimeout(() => {
            setVisible(false)
            setTimeout(() => onRemove(toast.id), 300)
        }, 4000)
        return () => clearTimeout(timer)
    }, [toast.id, onRemove])

    return (
        <div
            style={{
                display: 'flex', alignItems: 'flex-start', gap: '0.75rem',
                padding: '1rem 1.25rem', borderRadius: '0.75rem',
                background: 'white', border: '1px solid var(--color-border)',
                boxShadow: '0 10px 25px rgba(0,0,0,0.08)',
                opacity: visible ? 1 : 0,
                transform: visible ? 'translateX(0)' : 'translateX(100%)',
                transition: 'all 0.3s cubic-bezier(0.22, 1, 0.36, 1)',
                maxWidth: 380, width: '100%',
            }}
        >
            <div style={{ width: 32, height: 32, borderRadius: '50%', background: config.bg, display: 'flex', alignItems: 'center', justifyContent: 'center', flexShrink: 0 }}>
                <Icon size={16} style={{ color: config.color }} />
            </div>
            <div style={{ flex: 1, minWidth: 0 }}>
                <p style={{ fontWeight: 700, fontSize: '0.875rem', color: 'var(--color-text)' }}>{toast.title}</p>
                {toast.message && <p style={{ fontSize: '0.75rem', color: 'var(--color-muted)', marginTop: 2 }}>{toast.message}</p>}
            </div>
            <button onClick={() => { setVisible(false); setTimeout(() => onRemove(toast.id), 300) }} style={{ background: 'none', border: 'none', cursor: 'pointer', color: 'var(--color-muted)', padding: 2 }}>
                <X size={14} />
            </button>
        </div>
    )
}

export function ToastProvider({ children }: { children: React.ReactNode }) {
    const [toasts, setToasts] = useState<Toast[]>([])

    const addToast = useCallback((type: ToastType, title: string, message?: string) => {
        const id = Date.now().toString(36) + Math.random().toString(36).slice(2)
        setToasts(prev => [...prev, { id, type, title, message }])
    }, [])

    const removeToast = useCallback((id: string) => {
        setToasts(prev => prev.filter(t => t.id !== id))
    }, [])

    return (
        <ToastContext.Provider value={{ addToast }}>
            {children}
            <div style={{
                position: 'fixed', top: '1rem', right: '1rem', zIndex: 9999,
                display: 'flex', flexDirection: 'column', gap: '0.5rem',
                pointerEvents: 'none',
            }}>
                {toasts.map(toast => (
                    <div key={toast.id} style={{ pointerEvents: 'auto' }}>
                        <ToastItem toast={toast} onRemove={removeToast} />
                    </div>
                ))}
            </div>
        </ToastContext.Provider>
    )
}
