'use client'

import { useState, useEffect, useMemo } from 'react'
import { Lead } from '@/store/crmStore'

interface UserData {
    id: string
    nama: string
    email: string
    role: string
    noWa: string
}

// Roles that can see ALL leads
const SUPERVISOR_ROLES = ['super_admin', 'admin', 'spv_marketing']

/**
 * Hook for role-based lead access control.
 * 
 * Visibility rules:
 * - super_admin / admin / spv_marketing → see ALL leads
 * - sales_senior / sales_junior → see only leads where:
 *   - assignedSales matches their name (lead is assigned to them)
 *   - createdBy matches their name (they created the lead)
 */
export function useLeadAccess() {
    const [userData, setUserData] = useState<UserData | null>(null)

    useEffect(() => {
        try {
            const stored = localStorage.getItem('jabnet_user')
            if (stored) {
                setUserData(JSON.parse(stored))
            }
        } catch { }
    }, [])

    const userRole = userData?.role || ''
    const userName = userData?.nama || ''
    const userId = userData?.id || ''
    const isSupervisor = SUPERVISOR_ROLES.includes(userRole)

    /**
     * Filter leads based on current user's role and ownership.
     */
    const filterLeads = useMemo(() => {
        return (leads: Lead[]): Lead[] => {
            // Supervisors see everything
            if (isSupervisor) return leads

            // Sales users: see own leads + leads they created
            return leads.filter(lead =>
                lead.assignedSales === userName ||
                lead.createdBy === userName
            )
        }
    }, [isSupervisor, userName])

    return {
        userData,
        userRole,
        userName,
        userId,
        isSupervisor,
        filterLeads,
    }
}
