import { create } from 'zustand'
import { persist } from 'zustand/middleware'

// ── Types ──────────────────────────────────────────────
export interface Lead {
    id: string
    namaLengkap: string
    noWa: string
    noWaAlt?: string
    alamat: string
    rtRw?: string
    kelurahan: string
    kecamatan: string
    latitude?: number
    longitude?: number
    segmen: string
    sumberLead: string
    referralDari?: string
    paketInterest?: string
    pipelineStage: string
    pipelineStageDate: string
    lostReason?: string
    lostNotes?: string
    priority: string
    estimatedValue?: number
    nextFollowupDate?: string
    notes?: string
    assignedSales: string
    createdBy: string
    createdAt: string
    updatedAt: string
}

export interface Activity {
    id: string
    leadId: string
    leadName: string
    activityType: string
    activityDate: string
    result: string
    notes?: string
    sales: string
    area: string
}

export interface SalesUser {
    id: string
    nama: string
    email: string
    role: string
    noWa: string
    isActive: boolean
    targetPsb: number
    targetVisit: number
}

// ── Store ──────────────────────────────────────────────
interface CRMStore {
    leads: Lead[]
    activities: Activity[]
    users: SalesUser[]

    // Lead actions
    addLead: (lead: Omit<Lead, 'id' | 'createdAt' | 'updatedAt' | 'pipelineStageDate'>) => void
    updateLead: (id: string, data: Partial<Lead>) => void
    deleteLead: (id: string) => void
    moveLead: (id: string, newStage: string) => void

    // Activity actions
    addActivity: (activity: Omit<Activity, 'id'>) => void

    // User actions
    addUser: (user: Omit<SalesUser, 'id'>) => void
    updateUser: (id: string, data: Partial<SalesUser>) => void
    toggleUserActive: (id: string) => void
}

const generateId = () => Date.now().toString(36) + Math.random().toString(36).slice(2, 8)
const now = () => new Date().toISOString()

export const useCRMStore = create<CRMStore>()(
    persist(
        (set) => ({
            leads: [],
            activities: [],
            users: [],

            addLead: (leadData) => set((state) => ({
                leads: [{
                    ...leadData,
                    id: generateId(),
                    pipelineStageDate: now(),
                    createdAt: now(),
                    updatedAt: now(),
                }, ...state.leads],
            })),

            updateLead: (id, data) => set((state) => ({
                leads: state.leads.map(l => l.id === id ? { ...l, ...data, updatedAt: now() } : l),
            })),

            deleteLead: (id) => set((state) => ({
                leads: state.leads.filter(l => l.id !== id),
            })),

            moveLead: (id, newStage) => set((state) => ({
                leads: state.leads.map(l =>
                    l.id === id
                        ? { ...l, pipelineStage: newStage, pipelineStageDate: now(), updatedAt: now() }
                        : l
                ),
            })),

            addActivity: (activityData) => set((state) => ({
                activities: [{
                    ...activityData,
                    id: generateId(),
                }, ...state.activities],
            })),

            addUser: (userData) => set((state) => ({
                users: [{ ...userData, id: generateId() }, ...state.users],
            })),

            updateUser: (id, data) => set((state) => ({
                users: state.users.map(u => u.id === id ? { ...u, ...data } : u),
            })),

            toggleUserActive: (id) => set((state) => ({
                users: state.users.map(u => u.id === id ? { ...u, isActive: !u.isActive } : u),
            })),
        }),
        {
            name: 'jabnet-crm-store',
        }
    )
)
